/*
DirectWindowRenderer class
*/
#include "DirectWindowRenderer.h"
#include "colorspaces.h"
#include "MediaView.h"
#include "draw_window.h"
#include <Autolock.h>

#include <stdio.h>

DirectWindowRenderer::DirectWindowRenderer()
{
	bitmap = NULL;
	bitmapBounds = NULL;
	view = NULL;
	cs_index = -1;
}

DirectWindowRenderer::~DirectWindowRenderer()
{
	if(bitmap && view)
		Dispose();
}

const char *DirectWindowRenderer::ColorSpaceName() const
{
	if(cs_index == -1)
		return "none";
	return colspace[cs_index].name;
}

color_space DirectWindowRenderer::ColorSpace()
{
	if(cs_index == -1)
		return B_NO_COLOR_SPACE;
	return colspace[cs_index].colspace;
}

void *DirectWindowRenderer::Bits()
{
	if(!bitmap)
		return NULL;
	return bitmap->Bits();
}

status_t DirectWindowRenderer::LockBits()
{
	return bitmap->LockBits();
}

status_t DirectWindowRenderer::UnlockBits()
{
	bitmap->UnlockBits();
	return B_OK;
}
	
	// render the frame in buff, the next buffer to write to is returned (can be the same)
	// should block until the buffer returned is ready to be fulled
void *DirectWindowRenderer::Draw(void *buff)
{
	PlayerWindow *window;
	
	if(!bitmap || !view)
		return NULL;

	int bpp; // BYTES per pixel
	
	char *bbits; // bitmap bits
	char *sbits; // screen bits

	long bmw, bmh, bmbpr;
	long sbpr;
	
	
	long i,j;
	char *psrc, *pdst;

	window = (PlayerWindow *)view->Window();
	if(!window)
		return bitmap->Bits();
	if(window->connection_disabled)
		return bitmap->Bits();
	if(!window->connected)
		return bitmap->Bits();
	BAutolock l(window->direct_lock);

	if(!l.IsLocked())
		return bitmap->Bits();

	sbits = (char *)(window->m_db_info.bits);
	sbpr = window->m_db_info.bytes_per_row;
	bpp = (window->m_db_info.bits_per_pixel + 7)/ 8;
	
	bbits = (char *)bitmap->Bits();
	
	bmw = (long) bitmap->Bounds().right + 1;
	bmh = (long) bitmap->Bounds().bottom + 1;
	bmbpr = bitmap->BytesPerRow();

	DEBUG(">>>>>>>>>> DDraw() : (%ld,%ld)\n", bmw, bmh);
	psrc = bbits;
	pdst = sbits + ((long) window->m_db_info.window_bounds.left) * bpp + ((long) window->m_db_info.window_bounds.top) * sbpr;
	for(j=0; j<bmh; j++, psrc += bmbpr, pdst += sbpr) {
		memcpy(pdst, psrc, bmw * bpp);
//		memset(pdst, 255, bmw * bpp);
	}
	
	DEBUG(">>>>>>>>>> DDraw() end\n", bmw, bmh);


	return bitmap->Bits();
}

	// redraw last frame
status_t DirectWindowRenderer::ReDraw()
{
	if(!bitmap || !view)
		return B_ERROR;
	if (view->Window()->LockWithTimeout(50000) == B_OK) {

		view->Picture()->DrawBitmap(bitmap, view->VideoBounds());
//		view->Invalidate(view->Bounds());
//		view->Flush();
		view->Sync();
		view->Window()->Unlock();
	}


	return B_OK;
}
	
status_t DirectWindowRenderer::Install(MediaView *view, media_format *format)
{
	bool found = false;
	int i=0;
	
	if(!format)
		return B_ERROR;
	
	if(format->type != B_MEDIA_RAW_VIDEO/* && format->type != B_MEDIA_ENCODED_VIDEO*/)
		return B_ERROR;

	for(i=0; !found && i<NUM_CSPACES; i++)
		if(colspace[i].colspace == format->u.raw_video.display.format)
			found = true;
	
	if(!found)
		return B_ERROR;
	DUMP_FORMAT(format);

	i--;
	cs_index = i;
	
	found = false;
	
/*	bitmapBounds = new BRect (0.0, 
						   0.0, 
						   format->u.encoded_video.output.display.line_width - 1.0,
						   format->u.encoded_video.output.display.line_count - 1.0);
*/
	bitmapBounds = new BRect (0.0, 
						   0.0, 
						   format->u.raw_video.display.line_width - 1.0,
						   format->u.raw_video.display.line_count - 1.0);

	bitmap = new BBitmap(*bitmapBounds, B_BITMAP_ACCEPTS_VIEWS, colspace[cs_index].colspace);
	if(!bitmap || bitmap->InitCheck() != B_OK) {
		puts("not supported.");
		if(bitmap) delete bitmap;
		bitmap = NULL;
		return B_ERROR;
	}
	puts("supported.");

	this->view = view;

	if(view->Window()->LockWithTimeout(50000) == B_OK) {
		view->Picture()->Invalidate(view->Picture()->Bounds());
		view->Invalidate(view->Bounds());
		view->Flush();
		view->Window()->Unlock();
	}
	return B_OK;
}

status_t DirectWindowRenderer::Use()
{
	if(!view)
		return B_ERROR;
	
	if(view->Window()->LockWithTimeout(50000) == B_OK) {
//#ifdef USE_SUBVIEW
		view->Picture()->Invalidate(view->Picture()->Bounds());
//#endif
		view->Invalidate(view->Bounds());
		view->Flush();
		view->Window()->Unlock();
	}
	return B_OK;
}

status_t DirectWindowRenderer::Dispose()
{
	if(!bitmap)
		return B_ERROR;

	if(view && view->Window()!=NULL && view->Window()->Lock()) {
//		view->Picture()->ClearViewOverlay();
		view->Picture()->SetViewColor(B_TRANSPARENT_32_BIT);
/*		if(view->Window()->Lock()) {
			view->Invalidate(view->Bounds());
			view->Flush();
			view->Window()->Unlock();
		}*/
		view->Window()->Unlock();
	}
//	bitmap->UnlockBits();
	if(bitmap)
		delete bitmap;
	bitmap = NULL;

	return B_OK;
}

color_space DirectWindowRenderer::NextColorSpace(int *cookie) // cookie = 0 first
{
	if(*cookie >= NUM_CSPACES)
		return B_NO_COLOR_SPACE;
	return colspace[(*cookie)++].colspace;
}

BRect DirectWindowRenderer::Bounds()
{
	if(!bitmap)
		return BRect();
	return bitmap->Bounds();
}

int32 DirectWindowRenderer::BytesPerRow()
{
	if(!bitmap)
		return 0;
	return bitmap->BytesPerRow();
}
