/*
OverlayRenderer class
*/
#include "OverlayRenderer.h"
#include "colorspaces.h"
#include "MediaView.h"

#include <stdio.h>

OverlayRenderer::OverlayRenderer()
{
	drawBitmap = NULL;
	decodeBitmap = NULL;
	drawBounds = NULL;
	decodeBounds = NULL;
	view = NULL;
	horizOffset = 0;
	horizPadding = 0;
	vertOffset = 0;
	vertPadding = 0;
	offsetThreadId = -1;
	decodeSemaphore = 0;
	drawSemaphore = 0;
	cs_index = -1;
}

OverlayRenderer::~OverlayRenderer()
{
	if(view)
		Dispose();
}

const char *OverlayRenderer::ColorSpaceName() const
{
	if(cs_index == -1)
		return "none";
	return colspace[cs_index].name;
}

color_space OverlayRenderer::ColorSpace()
{
	if(cs_index == -1)
		return B_NO_COLOR_SPACE;
	return colspace[cs_index].colspace;
}

void *OverlayRenderer::Bits()
{
	if(!decodeBitmap)
		return NULL;
	return decodeBitmap->Bits();
}

status_t OverlayRenderer::LockBits()
{
	if (offsetThreadId>=0) {
		return decodeBitmap->LockBits();
	}
	return B_OK;
}

status_t OverlayRenderer::UnlockBits()
{
	if (offsetThreadId>=0) {
		decodeBitmap->UnlockBits();
	}
	return B_OK;
}
	
	// render the frame in buff, the next buffer to write to is returned (can be the same)
	// should block until the buffer returned is ready to be fulled
void *OverlayRenderer::Draw(void *buff)
{
	if (offsetThreadId>=0) {
		status_t status = B_OK;
		status = acquire_sem(decodeSemaphore);
		if (status == B_OK) {
			release_sem(drawSemaphore);
		}
	}
	if(!decodeBitmap)
		return NULL;
	return decodeBitmap->Bits();
}

status_t OverlayRenderer::TryColorSpace(MediaView * view, media_format * format, color_space cs) {
	uint32 decodeWidth = format->u.raw_video.display.line_width;
	uint32 decodeHeight = format->u.raw_video.display.line_count;

	overlay_restrictions r;
	BRect * dummyBounds = new BRect(0.0,0.0,decodeWidth - 1.0, decodeHeight - 1.0);
#ifdef HACK_BGA // ;-)
	BBitmap * dummyBitmap = new BBitmap(*dummyBounds,B_BITMAP_WILL_OVERLAY, cs);
#else
	BBitmap * dummyBitmap = new BBitmap(*dummyBounds,B_BITMAP_WILL_OVERLAY|B_BITMAP_RESERVE_OVERLAY_CHANNEL, cs);
#endif
	if(!dummyBitmap || dummyBitmap->InitCheck() != B_OK) {
		puts("not supported.");
		if (dummyBitmap) delete dummyBitmap;
		Dispose();
		return B_ERROR;
	}
	dummyBitmap->GetOverlayRestrictions(&r);
	delete dummyBitmap;
	dummyBitmap=NULL;
	
	printf("Checking overlay limits...\n");
	uint32 drawWidth = decodeWidth;
	uint32 drawHeight = decodeHeight;
	// adjust alignment
	if (r.source.horizontal_alignment!=0) {
		printf("Warning: horizontal alignment not compensated for.\n");
		printf("  Src horiz alignment  : %d\n",r.source.horizontal_alignment);
	}
	if (r.source.vertical_alignment!=0) {
		printf("Warning: vertical alignment not compensated for.\n");
		printf("  Src vert alignment   : %d\n",r.source.vertical_alignment);
	}
	if (r.source.width_alignment!=0) {
		printf("Checking width alignment.\n");
		printf("  Src width alignment  : %d\n",r.source.width_alignment+1);
		uint32 width=(r.source.width_alignment+1);
		printf("  Leftovers = %d\n",drawWidth % width);
		if (drawWidth%width>0) {
			horizPadding=width-drawWidth%width;
			printf("  Adding %d padding to enforce alignment.\n",horizPadding);
		} else {
			printf("  Alignment is good.\n");
		}
		drawWidth+=horizPadding;
	}
	if (r.source.height_alignment!=0) {
		printf("Warning: height alignment not compensated for.\n");
		printf("  Src height alignment : %d\n",r.source.height_alignment);
		drawHeight+=vertPadding;
	}
	// adjust size
	if (r.source.min_width>drawWidth) {
		printf("Warning: source width smaller than minimum for overlay.\n");
		printf("  %d < %d\n",drawWidth,r.source.min_width);
	}
	if (r.source.min_height>drawHeight) {
		printf("Warning: source height smaller than minimum for overlay.\n");
		printf("  %d < %d\n",drawHeight,r.source.min_height);
	}
	if (r.source.max_width<drawWidth) {
		printf("Warning: source width larger than maximum for overlay.\n");
		printf("  %d > %d\n",drawWidth,r.source.max_width);
	}
	if (r.source.max_height<drawHeight) {
		printf("Warning: source height larger than maximum for overlay.\n");
		printf("  %d > %d\n",drawHeight,r.source.max_height);
	}
	
	printf("Not checked:\n");
	printf("  Dst horiz alignment  : %08x\n",r.destination.horizontal_alignment);
	printf("  Dst vert alignment   : %08x\n",r.destination.vertical_alignment);
	printf("  Dst width alignment  : %08x\n",r.destination.width_alignment);
	printf("  Dst height alignment : %08x\n",r.destination.height_alignment);
	printf("  Dst min/max          : (%d,%d)/(%d,%d)\n",r.destination.min_width,r.destination.min_height,
														r.destination.max_width,r.destination.max_height);
	printf("  Min/max scaling      : (%f,%f)/(%f,%f)\n",r.min_width_scale,r.min_height_scale,
														r.max_width_scale,r.max_height_scale);
	
	if ((drawWidth == decodeWidth) && (drawHeight == decodeHeight)) {
		drawBounds = decodeBounds = new BRect (0.0, 0.0, decodeWidth - 1.0, decodeHeight - 1.0);
		drawBitmap = decodeBitmap = new BBitmap (*decodeBounds,B_BITMAP_WILL_OVERLAY|B_BITMAP_RESERVE_OVERLAY_CHANNEL, cs);
		if(!decodeBitmap || decodeBitmap->InitCheck() != B_OK) {
			puts("not supported.");
			Dispose();
			return B_ERROR;
		}
		puts("supported.");
		this->view = view;
	} else {
		printf("using offset bitmap.\n");
		printf("drawBounds = (%d,%d)\n",drawWidth,drawHeight);
		drawBounds = new BRect(0.0, 0.0, decodeWidth - 1.0, decodeHeight - 1.0);
		drawBitmap = new BBitmap(*drawBounds,B_BITMAP_WILL_OVERLAY|B_BITMAP_RESERVE_OVERLAY_CHANNEL, cs);
		if(!drawBitmap || drawBitmap->InitCheck() != B_OK) {
			puts("not supported - drawBitmap.");
			Dispose();
			return B_ERROR;
		}
		printf("decodeBounds = (%d,%d)\n",decodeWidth,decodeHeight);
		decodeBounds = new BRect(0.0, 0.0, decodeWidth - 1.0, decodeHeight - 1.0);
		decodeBitmap = new BBitmap(*decodeBounds,cs);
		if(!decodeBitmap || decodeBitmap->InitCheck() != B_OK) {
			puts("not supported - decodeBitmap.");
			Dispose();
			return B_ERROR;
		}
		offsetThreadId = spawn_thread((thread_func)(OverlayRenderer::offset), "offset", OFFSET_PRIORITY, this);
		if (offsetThreadId == 0) {
			puts("unable to start offset thread.");
			offsetThreadId = -1;
			Dispose();
			return B_ERROR;
		}
		puts("supported.");
		this->view = view;
		drawSemaphore = create_sem(0, "dblbuff_dec");
		decodeSemaphore = create_sem(0, "dblbuff_draw");
		resume_thread(offsetThreadId);
	}

	if(view->Window()->LockWithTimeout(50000) == B_OK) {
		view->Picture()->Invalidate(view->Picture()->Bounds());
		view->Invalidate(view->Bounds());
		view->Flush();
		view->Window()->Unlock();
	}
	return B_OK;
}
	
status_t OverlayRenderer::Install(MediaView *view, media_format *format)
{
	puts(">install");
	bool found = false;
	int i=0;
	
	if(!format)
		return B_ERROR;
	
	if(format->type != B_MEDIA_RAW_VIDEO/* && format->type != B_MEDIA_ENCODED_VIDEO*/)
		return B_ERROR;

	DUMP_FORMAT(format);
	
	for(i=0; !found && i<SUPP_OVL; i++)
		if(colspace[i].colspace == format->u.raw_video.display.format)
			found = true;
	
	if(!found)
		return B_ERROR;

	i--;
	cs_index = i;
	
	found = false;

	printf("Using overlay: %s (%04X)\n", colspace[cs_index].name, colspace[cs_index]);
	
	status_t status;
	status = TryColorSpace(view,format,colspace[i].colspace);
	if (status != B_OK) return status;

	puts("<install");
	return B_OK;
}

status_t OverlayRenderer::Use()
{
	puts(">use");

	if(!view)
		return B_ERROR;
	
	rgb_color key;
	if(view && view->Window()!=NULL && view->Window()->LockWithTimeout(50000) == B_OK) {
		view->Picture()->SetViewOverlay(drawBitmap,*drawBounds,view->VideoBounds(),&key,B_FOLLOW_ALL,
			B_OVERLAY_FILTER_HORIZONTAL|B_OVERLAY_FILTER_VERTICAL);
		view->Picture()->SetViewColor(key);
//#ifdef USE_SUBVIEW
		view->Picture()->Invalidate(view->Picture()->Bounds());
//#endif
		view->Invalidate(view->Bounds());
		view->Flush();
		view->Window()->Unlock();
	}

	int32 * drawBits = (int32*)(drawBitmap->Bits());
	int32 * drawEnd = (int32*)(((char*)drawBits)+drawBitmap->BitsLength());
	for( ; drawBits<drawEnd ; drawBits++) {
		*(drawBits)=(int32)0x80008000;
	}

	if (offsetThreadId >= 0) release_sem(decodeSemaphore);
	puts("<use");
	return B_OK;
}

status_t OverlayRenderer::Dispose()
{
	puts(">dispose");

	status_t err;
//	if (offsetThreadId >= 0) wait_for_thread(offsetThreadId, &err);
	offsetThreadId = -1;
	if (decodeSemaphore) delete_sem(decodeSemaphore);
	decodeSemaphore = 0;
	if (drawSemaphore) delete_sem(drawSemaphore);
	drawSemaphore = 0;

	if(view && view->Window()!=NULL && view->Window()->Lock()) {
		view->Picture()->ClearViewOverlay();
		view->Picture()->SetViewColor(B_TRANSPARENT_32_BIT);
/*		if(view->Window()->Lock()) {
			view->Invalidate(view->Bounds());
			view->Flush();
			view->Window()->Unlock();
		}*/
		view->Window()->Unlock();
	}
//	drawBitmap->UnlockBits();
	view = NULL;
	
	// check for sharing
	if (decodeBitmap==drawBitmap) decodeBitmap=0;
	if (decodeBounds==drawBounds) decodeBounds=0;
	
	// free it up
	if (decodeBitmap) delete decodeBitmap;
	decodeBitmap = NULL;
	if (decodeBounds) delete decodeBounds;
	decodeBounds = NULL;
	if (drawBitmap) delete drawBitmap;
	drawBitmap = NULL;
	if (drawBounds) delete drawBounds;
	drawBounds = NULL;

	puts("<dispose");
	return B_OK;
}

color_space OverlayRenderer::NextColorSpace(int *cookie) // cookie = 0 first
{
	if(*cookie >= SUPP_OVL)
		return B_NO_COLOR_SPACE;
	return colspace[(*cookie)++].colspace;
}

BRect OverlayRenderer::Bounds()
{
	if(!decodeBitmap)
		return BRect();
	return decodeBitmap->Bounds();
}

int32 OverlayRenderer::BytesPerRow()
{
	if(!decodeBitmap)
		return 0;
	return decodeBitmap->BytesPerRow();
}

int32 OverlayRenderer::offset(void * argument) 
{
	OverlayRenderer *renderer;
	MediaView *view;
	
	renderer = (OverlayRenderer*)argument;
	view = renderer->view;

	while(acquire_sem(renderer->drawSemaphore) == B_OK) {
		if (!renderer->drawBitmap || !view) break;
		
		renderer->decodeBitmap->LockBits();
		renderer->drawBitmap->LockBits();

		int32 * decodeBits = (int32*)(renderer->decodeBitmap->Bits());
		int32 * decodeEnd = (int32*)(((char*)decodeBits)+renderer->decodeBitmap->BitsLength());
		int32 * drawBits = (int32*)(renderer->drawBitmap->Bits());
		int32 * drawEnd = (int32*)(((char*)drawBits)+renderer->drawBitmap->BitsLength());
		int32 drawBytes = renderer->drawBitmap->BytesPerRow();
		int32 decodeBytes = renderer->decodeBitmap->BytesPerRow();
		
		int32 margin=(drawBytes-decodeBytes)/4;
		int32 width=decodeBytes/4;
		
		int32 paddingIndex = 1;
		//drawBits+=margin/2;

	//	this version seems slower?
		for( ; decodeBits<decodeEnd && drawBits<drawEnd ; ) {
			memcpy(drawBits,decodeBits,width*4);
			drawBits+=margin+width;
			decodeBits+=width;
		}

//		for( ; decodeBits<decodeEnd && drawBits<drawEnd ; decodeBits++, drawBits++) {
//			*(drawBits)=*(decodeBits);
//			if (paddingIndex++ == width) {
//				drawBits+=margin;
//				paddingIndex=1;
//			}
//		}

/* backup copy with uchars
		uchar * decodeBits = (uchar*)(renderer->decodeBitmap->Bits());
		uchar * decodeEnd = (uchar*)(((char*)decodeBits)+renderer->decodeBitmap->BitsLength());
		uchar * drawBits = (uchar*)(renderer->drawBitmap->Bits());
		uchar * drawEnd = (uchar*)(((char*)drawBits)+renderer->drawBitmap->BitsLength());
		int32 decodeWidth = renderer->drawBounds->IntegerWidth();
		int32 drawBytes = renderer->drawBitmap->BytesPerRow();
		int32 decodeBytes = renderer->decodeBitmap->BytesPerRow();
		
		int paddingIndex = 0;
		drawBits+=renderer->horizPadding;
		for( ; decodeBits<decodeEnd ; drawBits++, decodeBits++) {
			if (paddingIndex++ > decodeWidth*2-renderer->horizPadding*2) {
				drawBits+=renderer->horizPadding*2;
				paddingIndex=0;
			}
			*(drawBits)=*(decodeBits);
			if (drawBits>drawEnd) break; // don't go off the end
		}
*/
		renderer->decodeBitmap->UnlockBits();
		renderer->drawBitmap->UnlockBits();
		release_sem(renderer->decodeSemaphore);
	}
	return B_OK;
}
