/*
DoubleBufferRenderer class
*/
#include "DoubleBufferRenderer.h"
#include "colorspaces.h"
#include "MediaView.h"
#include <SupportDefs.h>

#include <stdio.h>

DoubleBufferRenderer::DoubleBufferRenderer()
{
	drawer = -1;
	decode_sem = 0;
	draw_sem = 0;
	droppedframes=0;
	bitmap[0] = NULL;
	bitmap[1] = NULL;
	bitmapBounds = NULL;
	bm_sz = 0;
	view = NULL;
	cs_index = -1;
}

DoubleBufferRenderer::~DoubleBufferRenderer()
{
	if(view)
		Dispose();
}

const char *DoubleBufferRenderer::ColorSpaceName() const
{
	if(cs_index == -1)
		return "none";
	return colspace[cs_index].name;
}

color_space DoubleBufferRenderer::ColorSpace()
{
	if(cs_index == -1)
		return B_NO_COLOR_SPACE;
	return colspace[cs_index].colspace;
}

void *DoubleBufferRenderer::Bits()
{
	if(!bitmap[BM_DECODE])
		return NULL;
	return bitmap[BM_DECODE]->Bits();
}

status_t DoubleBufferRenderer::LockBits()
{
	return bitmap[BM_DECODE]->LockBits();
}

status_t DoubleBufferRenderer::UnlockBits()
{
	bitmap[BM_DECODE]->UnlockBits();
	return B_OK;
}
	
	// render the frame in buff, the next buffer to write to is returned (can be the same)
	// should block until the buffer returned is ready to be fulled
void *DoubleBufferRenderer::Draw(void *buff)
{
	status_t err = B_OK;
////	if((err=acquire_sem_etc(decode_sem, 1, B_RELATIVE_TIMEOUT, 500000)) == B_OK) {
	if((err=acquire_sem(decode_sem)) == B_OK) {
////		if(!bitmap || !view)
////			break;

	bitmap[BM_DECODE]->LockBits();
	bitmap[BM_DRAW]->LockBits();
	memcpy(bitmap[BM_DRAW]->Bits(), bitmap[BM_DECODE]->Bits(), bm_sz); // we only copy the Bits(), not the BBitmaps :)))) (I should go to bed now)
	bitmap[BM_DRAW]->UnlockBits();
	bitmap[BM_DECODE]->UnlockBits();
		
		release_sem(draw_sem);
//	} else if(err == B_TIMEOUT) {
//		droppedframes++;
	}


	return bitmap[BM_DECODE]->Bits();
}

	// redraw last frame
status_t DoubleBufferRenderer::ReDraw()
{
	if(!bitmap[0] || !bitmap[1] || !view)
		return B_ERROR;
 	if (view->Window()->LockWithTimeout(50000) == B_OK) {

		view->Picture()->DrawBitmapAsync(bitmap[BM_DRAW], view->VideoBounds());
//		view->Invalidate(view->Bounds());
//		view->Flush();
		view->Picture()->Sync();
		view->Window()->Unlock();
	}

	return B_OK;
}
	
status_t DoubleBufferRenderer::Install(MediaView *view, media_format *format)
{
	bool found = false;
	int i=0;
	
	droppedframes=0;

	if(!format)
		return B_ERROR;
	
	if(format->type != B_MEDIA_RAW_VIDEO/* && format->type != B_MEDIA_ENCODED_VIDEO*/)
		return B_ERROR;

	for(i=0; !found && i<NUM_CSPACES; i++)
		if(colspace[i].colspace == format->u.raw_video.display.format)
			found = true;
	
	if(!found)
		return B_ERROR;
	DUMP_FORMAT(format);

	i--;
	cs_index = i;
	
	found = false;
	
/*	bitmapBounds = new BRect (0.0, 
						   0.0, 
						   format->u.encoded_video.output.display.line_width - 1.0,
						   format->u.encoded_video.output.display.line_count - 1.0);
*/
	bitmapBounds = new BRect (0.0, 
						   0.0, 
						   format->u.raw_video.display.line_width - 1.0,
						   format->u.raw_video.display.line_count - 1.0);

	bitmap[0] = new BBitmap(*bitmapBounds, B_BITMAP_ACCEPTS_VIEWS, colspace[cs_index].colspace);
	if(!bitmap[0] || bitmap[0]->InitCheck() != B_OK) {
		puts("not supported. (1)");
		if(bitmap[0]) delete bitmap[0];
		bitmap[0] = NULL;
		puts("not supported. (1)");
		return B_ERROR;
	}
	bitmap[1] = new BBitmap(*bitmapBounds, B_BITMAP_ACCEPTS_VIEWS, colspace[cs_index].colspace);
	if(!bitmap[1] || bitmap[1]->InitCheck() != B_OK) {
		puts("not supported. (2)");
		if(bitmap[0]) delete bitmap[0];
		bitmap[0] = NULL;
		if(bitmap[1]) delete bitmap[1];
		bitmap[1] = NULL;
		return B_ERROR;
	}
	bm_sz = bitmap[BM_DRAW]->BitsLength();
	
	puts("supported.");

	this->view = view;

	draw_sem = create_sem(0, "dblbuff_dec");
	decode_sem = create_sem(0, "dblbuff_draw");

	drawer = spawn_thread(DoubleBufferRenderer::DrawerThread, "drawer", 15, this);
	if(drawer == 0) {
		if(bitmap[0]) delete bitmap[0];
		bitmap[0] = NULL;
		if(bitmap[1]) delete bitmap[1];
		bitmap[1] = NULL;
		delete_sem(draw_sem);
		delete_sem(decode_sem);
		return B_ERROR;
	}
	
	resume_thread(drawer);

	if(view->Window()->LockWithTimeout(50000) == B_OK) {
		view->Picture()->Invalidate(view->Picture()->Bounds());
		view->Invalidate(view->Bounds());
		view->Flush();
		view->Window()->Unlock();
	}
	return B_OK;
}

status_t DoubleBufferRenderer::Use()
{
	if(!view)
		return B_ERROR;
	release_sem(decode_sem);
//	release_sem(decode_sem);
	if(view->Window()->LockWithTimeout(50000) == B_OK) {
//#ifdef USE_SUBVIEW
		view->Picture()->Invalidate(view->Picture()->Bounds());
//#endif
//		view->Invalidate(view->Bounds());
		view->Flush();
		view->Window()->Unlock();
	}
	return B_OK;
}

status_t DoubleBufferRenderer::Dispose()
{
	status_t err;
	
	if(!bitmap[0] || !bitmap[1])
		return B_ERROR;
	if(decode_sem)
		delete_sem(decode_sem);
	decode_sem=0;
	if(draw_sem)
		delete_sem(draw_sem);
	draw_sem=0;
	if(drawer != -1)
		wait_for_thread(drawer, &err);
	drawer = -1;
	if(view && view->Window()!=NULL && view->Window()->Lock()) {
//		view->Picture()->ClearViewOverlay();
		view->Picture()->SetViewColor(B_TRANSPARENT_32_BIT);
/*		if(view->Window()->Lock()) {
			view->Invalidate(view->Bounds());
			view->Flush();
			view->Window()->Unlock();
		}*/
		view->Window()->Unlock();
	}
//	bitmap->UnlockBits();
	if(bitmap[0])
		delete bitmap[0];
	bitmap[0] = NULL;
	if(bitmap[1])
		delete bitmap[1];
	bitmap[1] = NULL;

	return B_OK;
}

color_space DoubleBufferRenderer::NextColorSpace(int *cookie) // cookie = 0 first
{
	if(*cookie >= NUM_CSPACES)
		return B_NO_COLOR_SPACE;
	return colspace[(*cookie)++].colspace;
}

BRect DoubleBufferRenderer::Bounds()
{
	if(!bitmap[0])
		return BRect();
	return bitmap[0]->Bounds();
}

int32 DoubleBufferRenderer::BytesPerRow()
{
	if(!bitmap[0])
		return 0;
	return bitmap[0]->BytesPerRow();
}

int32 DoubleBufferRenderer::DrawerThread(void *arg)
{
	DoubleBufferRenderer *r;
	MediaView *view;
	
	r = (DoubleBufferRenderer *)arg;
	view = r->view;
	
	while(acquire_sem(r->draw_sem) == B_OK) {
		if(!r->bitmap[BM_DRAW] || !view)
			break;
		if (view->Window()->LockWithTimeout(50000) == B_OK) {
			view->Picture()->DrawBitmapAsync(r->bitmap[BM_DRAW], view->VideoBounds());
//			view->Invalidate(view->Bounds());
//			view->Flush();
			view->Picture()->Sync();
			view->Window()->Unlock();
		}
		release_sem(r->decode_sem);
	}

	return B_OK;
}
