/*
MagicOverlayRenderer class
*/
#include "MagicOverlayRenderer.h"
#include "colorspaces.h"
#include "MediaView.h"
#include "rgb_to_ycbcr.h"
#include <alloc.h>
#include <stdio.h>

#ifndef uchar
typedef unsigned char uchar;
#endif

MagicOverlayRenderer::MagicOverlayRenderer()
{
	drawBitmap = NULL;
	decodeBitmap = NULL;
	drawBounds = NULL;
	decodeBounds = NULL;
	view = NULL;
	horizOffset = 0;
	horizPadding = 0;
	vertOffset = 0;
	vertPadding = 0;
	convertThreadId = -1;
	droppedframes=0;
	cs_index = -1;
	SetupYCbCr422Tables();
}

MagicOverlayRenderer::~MagicOverlayRenderer()
{
	if(view)
		Dispose();
	// free the YCbCr422Tables
	if(lt_r_y)
		free(lt_r_y);
	lt_r_y = NULL;
	if(lt_g_y)
		free(lt_g_y);
	lt_g_y = NULL;
	if(lt_b_y)
		free(lt_b_y);
	lt_b_y = NULL;
	if(lt_r_cb)
		free(lt_r_cb);
	lt_r_cb = NULL;
	if(lt_g_cb)
		free(lt_g_cb);
	lt_g_cb = NULL;
	if(lt_b_cb)
		free(lt_b_cb);
	lt_b_cb = NULL;
	if(lt_g_cr)
		free(lt_g_cr);
	lt_g_cr = NULL;
	if(lt_b_cr)
		free(lt_b_cr);
	lt_b_cr = NULL;
}

const char *MagicOverlayRenderer::ColorSpaceName() const
{
	if(cs_index == -1)
		return "none";
	return colspace[cs_index].name;
}

color_space MagicOverlayRenderer::ColorSpace()
{
	if(cs_index == -1)
		return B_NO_COLOR_SPACE;
	return colspace[cs_index].colspace;
}

void *MagicOverlayRenderer::Bits()
{
	if(!decodeBitmap)
		return NULL;
	return decodeBitmap->Bits();
}

status_t MagicOverlayRenderer::LockBits()
{
	return decodeBitmap->LockBits();
}

status_t MagicOverlayRenderer::UnlockBits()
{
	decodeBitmap->UnlockBits();
	return B_OK;
}
	
	// render the frame in buff, the next buffer to write to is returned (can be the same)
	// should block until the buffer returned is ready to be fulled
void *MagicOverlayRenderer::Draw(void *buff)
{
	status_t status = B_OK;
////	if((err=acquire_sem_etc(decode_sem, 1, B_RELATIVE_TIMEOUT, 500000)) == B_OK) {
	status = acquire_sem(decodeSemaphore);
	if (status == B_OK) {
////		if(!bitmap || !view)
////			break;
/*
	bitmap[BM_DECODE]->LockBits();
	bitmap[BM_DRAW]->LockBits();
	memcpy(bitmap[BM_DRAW]->Bits(), bitmap[BM_DECODE]->Bits(), bm_sz); // we only copy the Bits(), not the BBitmaps :)))) (I should go to bed now)
	bitmap[BM_DRAW]->UnlockBits();
	bitmap[BM_DECODE]->UnlockBits();
*/		
		release_sem(drawSemaphore);
//	} else if(err == B_TIMEOUT) {
//		droppedframes++;
	}
	if (!decodeBitmap)
		return NULL;
	return decodeBitmap->Bits();
}

void MagicOverlayRenderer::SetupYCbCr422Tables() {
	puts(">YCbCr422Tables");
	lt_r_y = (int32 *)malloc(TABLE_SIZE * SIZEOF(INT32));
	lt_g_y = (int32 *)malloc(TABLE_SIZE * SIZEOF(INT32));
	lt_b_y = (int32 *)malloc(TABLE_SIZE * SIZEOF(INT32));
	lt_r_cb = (int32 *)malloc(TABLE_SIZE * SIZEOF(INT32));
	lt_g_cb = (int32 *)malloc(TABLE_SIZE * SIZEOF(INT32));
	lt_b_cb = (int32 *)malloc(TABLE_SIZE * SIZEOF(INT32));
	lt_r_cr = lt_b_cb; // same as lt_b_cb
	lt_g_cr = (int32 *)malloc(TABLE_SIZE * SIZEOF(INT32));
	lt_b_cr = (int32 *)malloc(TABLE_SIZE * SIZEOF(INT32));
	for (int i = 0; i <= MAXSAMPLE; i++) {
#ifdef NEVRAX_EQS
		lt_r_y[i] = FIX(0.29900) * i;
		lt_g_y[i] = FIX(0.58700) * i;
		lt_b_y[i] = FIX(0.11400) * i     + ONE_HALF;
		lt_r_cb[i] = (-FIX(0.16874)) * i;
		lt_g_cb[i] = (-FIX(0.33126)) * i;
		/* We use a rounding fudge-factor of 0.5-epsilon for Cb and Cr.
		 * This ensures that the maximum output will round to MAXJSAMPLE
		 * not MAXJSAMPLE+1, and thus that we don't have to range-limit.
		 */
		lt_b_cb[i] = FIX(0.50000) * i    + CBCR_OFFSET + ONE_HALF-1;
		/*  B=>Cb and R=>Cr tables are the same
		    rgb_ycc_tab[i+R_CR_OFF] = FIX(0.50000) * i    + CBCR_OFFSET + ONE_HALF-1;
		 */
		lt_g_cr[i] = (-FIX(0.41869)) * i;
		lt_b_cr[i] = (-FIX(0.08131)) * i;
#else
		lt_r_y[i] = FIX(0.25700) * i;
		lt_g_y[i] = FIX(0.50400) * i;
		lt_b_y[i] = FIX(0.09800) * i     + ONE_HALF + Y_OFFSET;
		lt_r_cb[i] = (-FIX(0.14800)) * i;
		lt_g_cb[i] = (-FIX(0.29100)) * i;
		/* We use a rounding fudge-factor of 0.5-epsilon for Cb and Cr.
		 * This ensures that the maximum output will round to MAXJSAMPLE
		 * not MAXJSAMPLE+1, and thus that we don't have to range-limit.
		 */
		lt_b_cb[i] = FIX(0.43900) * i    + CBCR_OFFSET + ONE_HALF-1;
		/*  B=>Cb and R=>Cr tables are the same
		    rgb_ycc_tab[i+R_CR_OFF] = FIX(0.50000) * i    + CBCR_OFFSET + ONE_HALF-1;
		 */
		lt_g_cr[i] = (-FIX(0.36800)) * i;
		lt_b_cr[i] = (-FIX(0.07100)) * i;
#endif
	}
	puts("<YCbCr422Tables");
}

status_t MagicOverlayRenderer::TryColorSpace(MediaView * view, media_format * format, color_space cs) {
	uint32 decodeWidth = format->u.raw_video.display.line_width;
	uint32 decodeHeight = format->u.raw_video.display.line_count;

	overlay_restrictions r;
	BRect * dummyBounds = new BRect(0.0,0.0,decodeWidth - 1.0, decodeHeight - 1.0);
#ifdef HACK_BGA // ;-)
	BBitmap * dummyBitmap = new BBitmap(*dummyBounds,B_BITMAP_WILL_OVERLAY, cs);
#else
	BBitmap * dummyBitmap = new BBitmap(*dummyBounds,B_BITMAP_WILL_OVERLAY|B_BITMAP_RESERVE_OVERLAY_CHANNEL, cs);
#endif
	if(!dummyBitmap || dummyBitmap->InitCheck() != B_OK) {
		puts("not supported. (dummy failure)");
		Dispose();
		if (dummyBitmap) delete dummyBitmap;
		return B_ERROR;
	}
	dummyBitmap->GetOverlayRestrictions(&r);
	delete dummyBitmap;
	dummyBitmap=NULL;
	
	printf("Checking overlay limits...\n");
	uint32 drawWidth = decodeWidth;
	uint32 drawHeight = decodeHeight;
	// adjust alignment
	if (r.source.horizontal_alignment!=0) {
		printf("Warning: horizontal alignment not compensated for.\n");
		printf("  Src horiz alignment  : %d\n",r.source.horizontal_alignment);
	}
	if (r.source.vertical_alignment!=0) {
		printf("Warning: vertical alignment not compensated for.\n");
		printf("  Src vert alignment   : %d\n",r.source.vertical_alignment);
	}
	if (r.source.width_alignment!=0) {
		printf("Checking width alignment.\n");
		printf("  Src width alignment  : %d\n",r.source.width_alignment+1);
		uint32 width=(r.source.width_alignment+1);
		printf("  Leftovers = %d\n",drawWidth % width);
		if (drawWidth%width>0) {
			horizPadding=width-drawWidth%width;
			printf("  Adding %d padding to enforce alignment.\n",horizPadding);
		} else {
			printf("  Alignment is good.\n");
		}
		drawWidth+=horizPadding;
	}
	if (r.source.height_alignment!=0) {
		printf("Warning: height alignment not compensated for.\n");
		printf("  Src height alignment : %d\n",r.source.height_alignment);
		drawHeight+=vertPadding;
	}
	// adjust size
	if (r.source.min_width>drawWidth) {
		printf("Warning: source width smaller than minimum for overlay.\n");
		printf("  %d < %d\n",drawWidth,r.source.min_width);
	}
	if (r.source.min_height>drawHeight) {
		printf("Warning: source height smaller than minimum for overlay.\n");
		printf("  %d < %d\n",drawHeight,r.source.min_height);
	}
	if (r.source.max_width<drawWidth) {
		printf("Warning: source width larger than maximum for overlay.\n");
		printf("  %d > %d\n",drawWidth,r.source.max_width);
	}
	if (r.source.max_height<drawHeight) {
		printf("Warning: source height larger than maximum for overlay.\n");
		printf("  %d > %d\n",drawHeight,r.source.max_height);
	}
	
	printf("Not checked:\n");
	printf("  Dst horiz alignment  : %08x\n",r.destination.horizontal_alignment);
	printf("  Dst vert alignment   : %08x\n",r.destination.vertical_alignment);
	printf("  Dst width alignment  : %08x\n",r.destination.width_alignment);
	printf("  Dst height alignment : %08x\n",r.destination.height_alignment);
	printf("  Dst min/max          : (%d,%d)/(%d,%d)\n",r.destination.min_width,r.destination.min_height,
														r.destination.max_width,r.destination.max_height);
	printf("  Min/max scaling      : (%f,%f)/(%f,%f)\n",r.min_width_scale,r.min_height_scale,
														r.max_width_scale,r.max_height_scale);
	
	if ((drawWidth != decodeWidth) && (drawHeight != decodeHeight)) {
		printf("using offset bitmap.\n");
	}
	
	printf("decodeBounds = (%d,%d)\n",decodeWidth,decodeHeight);
	decodeBounds = new BRect(0.0, 0.0, decodeWidth - 1.0, decodeHeight - 1.0);
	decodeBitmap = new BBitmap(*decodeBounds,B_BITMAP_ACCEPTS_VIEWS,colspace[cs_index].colspace);
	if(!decodeBitmap || decodeBitmap->InitCheck() != B_OK) {
		puts("not supported - decodeBitmap.");
		Dispose();
		return B_ERROR;
	}
	printf("drawBounds = (%d,%d)\n",drawWidth,drawHeight);
	drawBounds = new BRect(0.0, 0.0, decodeWidth - 1.0, decodeHeight - 1.0);
	drawBitmap = new BBitmap(*drawBounds,B_BITMAP_WILL_OVERLAY|B_BITMAP_RESERVE_OVERLAY_CHANNEL, cs);
	if(!drawBitmap || drawBitmap->InitCheck() != B_OK) {
		puts("not supported - drawBitmap.");
		Dispose();
		return B_ERROR;
	}
	this->view = view;
	drawSemaphore = create_sem(0, "dblbuff_dec");
	decodeSemaphore = create_sem(0, "dblbuff_draw");

	thread_func convert_func = 0;
	if (cs == B_RGB16) {
		convert_func = MagicOverlayRenderer::DrawerThreadRGB16;
	} else if (cs == B_YCbCr422) {
		convert_func = MagicOverlayRenderer::DrawerThreadYCbCr422;
	} else {
		puts("no drawer thread for this colorspace.");
		Dispose();
		return B_ERROR;
	}
	convertThreadId = spawn_thread(convert_func,"drawer", DRAWER_PRIORITY, this);
	if (convertThreadId == 0) {
		puts("unable to start drawer thread.");
		convertThreadId = -1;
		Dispose();
		return B_ERROR;
	}		
	puts("supported.");
	resume_thread(convertThreadId);

	if(view->Window()->LockWithTimeout(50000) == B_OK) {
		view->Picture()->Invalidate(view->Picture()->Bounds());
		view->Invalidate(view->Bounds());
		view->Flush();
		view->Window()->Unlock();
	}
	
	return B_OK;
}

status_t MagicOverlayRenderer::Install(MediaView *view, media_format *format)
{
	puts(">install");
	bool found = false;

	droppedframes=0;

	if(!format)
		return B_ERROR;
	
	if(format->type != B_MEDIA_RAW_VIDEO/* && format->type != B_MEDIA_ENCODED_VIDEO*/)
		return B_ERROR;

	DUMP_FORMAT(format);
	
	if(format->u.raw_video.display.format != B_RGB32)
		return B_ERROR; // only RGB32 -> YCbCr (for now)
	
	int i;
	for(i=0; !found && i<SUPP_OVL; i++)
		if(colspace[i].colspace == format->u.raw_video.display.format)
			found = true;
	
	if(!found)
		return B_ERROR;

	i--;
	cs_index = i;
	
	found = false;

	// first try B_RGB16
	status_t status;
	puts("trying B_RGB16 overlay");
	status = TryColorSpace(view,format,B_RGB16);
	if (status == B_OK) {
		puts("using B_RGB16 overlay");
	} else {
		puts("B_RGB16 not supported. (2)");
		// it failed... try B_YCbCr422
		puts("trying B_YCbCr422");
		status = TryColorSpace(view,format,B_YCbCr422);
		if (status == B_OK) {
			puts("using B_YCbCr422 overlay");
		} else {
			return status;
		}
	}

	puts("<install");
	return B_OK;
}

status_t MagicOverlayRenderer::Use()
{
	int i;

	puts(">use");

	if(!view)
		return B_ERROR;
	
	rgb_color key;
	if(view && view->Window()!=NULL && view->Window()->LockWithTimeout(50000) == B_OK) {
		view->Picture()->SetViewOverlay(drawBitmap,*drawBounds,view->VideoBounds(),&key,B_FOLLOW_ALL,
			B_OVERLAY_FILTER_HORIZONTAL|B_OVERLAY_FILTER_VERTICAL);
		view->Picture()->SetViewColor(key);
//#ifdef USE_SUBVIEW
		view->Picture()->Invalidate(view->Picture()->Bounds());
//#endif
		view->Invalidate(view->Bounds());
		view->Flush();
		view->Window()->Unlock();
	}
	
	int32 * drawBits = (int32*)(drawBitmap->Bits());
	int32 * drawEnd = (int32*)(((char*)drawBits)+drawBitmap->BitsLength());
	for( ; drawBits<drawEnd ; drawBits++) {
		*(drawBits)=(int32)0x80008000;
//		*(drawBits)=(int32)0x40404040; // for junk
	}

	release_sem(decodeSemaphore);
	puts("<use");
	return B_OK;
}

status_t MagicOverlayRenderer::Dispose()
{
	puts(">dispose");
	
	status_t err;
//	if (convertThreadId >= 0) wait_for_thread(convertThreadId, &err);
	convertThreadId = -1;

	if (decodeSemaphore) delete_sem(decodeSemaphore);
	decodeSemaphore = 0;
	if (drawSemaphore) delete_sem(drawSemaphore);
	drawSemaphore = 0;
	
	if(view && view->Window()!=NULL && view->Window()->Lock()) {
		view->Picture()->ClearViewOverlay();
		view->Picture()->SetViewColor(B_TRANSPARENT_32_BIT);
/*		if(view->Window()->Lock()) {
			view->Invalidate(view->Bounds());
			view->Flush();
			view->Window()->Unlock();
		}*/
		view->Window()->Unlock();
	}
	view = NULL;
//	bitmap->UnlockBits();

	// check for sharing
	if (decodeBitmap==drawBitmap) decodeBitmap=0;
	if (decodeBounds==drawBounds) decodeBounds=0;

	// free it up
	if (decodeBitmap) delete decodeBitmap;
	decodeBitmap = NULL;
	if (decodeBounds) delete decodeBounds;
	decodeBounds = NULL;
	if (drawBitmap) delete drawBitmap;
	drawBitmap = NULL;
	if (drawBounds) delete drawBounds;
	drawBounds = NULL;

	puts("<dispose");
	return B_OK;
}

color_space MagicOverlayRenderer::NextColorSpace(int *cookie) // cookie = 0 first
{
	if(*cookie >= SUPP_OVL)
		return B_NO_COLOR_SPACE;
	return colspace[(*cookie)++].colspace;
}

BRect MagicOverlayRenderer::Bounds()
{
	if(!decodeBitmap)
		return BRect();
	return decodeBitmap->Bounds();
}

int32 MagicOverlayRenderer::BytesPerRow()
{
	if(!decodeBitmap)
		return 0;
	return decodeBitmap->BytesPerRow();
}


int32 MagicOverlayRenderer::DrawerThreadRGB16(void *arg)
{
	MagicOverlayRenderer *renderer;
	MediaView *view;
	
	renderer = (MagicOverlayRenderer *)arg;
	view = renderer->view;

//	register long cnt;
	register int r, g, b;
	register int r1, g1, b1;
	register uchar y0, y1, cb, cr;
	
	register int32 ov_long;
	register int32 rgb_long;
	
//	printf("RGB_SZ=%ld drawbm_sz=%ld\n", renderer->rgb_sz, renderer->drawbm_sz);
	
	while(acquire_sem(renderer->drawSemaphore) == B_OK) {
		if(!renderer->drawBitmap || !view)
			break;
//		puts("MOR - got lock");
			
		renderer->decodeBitmap->LockBits();
		renderer->drawBitmap->LockBits();
		
		int32 * decodeBits = (int32*)(renderer->decodeBitmap->Bits());
		int32 * decodeEnd = (int32*)(((char*)decodeBits)+renderer->decodeBitmap->BitsLength());
		int32 * drawBits = (int32*)(renderer->drawBitmap->Bits());
		int32 * drawEnd = (int32*)(((char*)drawBits)+renderer->drawBitmap->BitsLength());
		int32 drawBytes = renderer->drawBitmap->BytesPerRow();
		int32 decodeBytes = renderer->decodeBitmap->BytesPerRow()/2;
		
		int32 margin=(drawBytes-decodeBytes)/4;
		int32 width=decodeBytes/4;
		
		int32 paddingIndex = 1;
//		drawBits+=margin/2;

		uchar * rgbplane = (uchar*)decodeBits;
		uchar * rgb_end = (uchar*)decodeEnd;
		for( ; rgbplane<rgb_end && drawBits<drawEnd ; rgbplane+=4, drawBits++) {
			b = *(rgbplane);
			g = *(rgbplane+1);
			r = *(rgbplane+2);
			
			// from GraphicsDefs.h
			// B_RGB16 =                       0x0005,
			// MSB                      LSB
			// G[2:0],B[4:0]  R[4:0],G[5:3]
	//		ov_long = (((((long)(g>>2)) & 0x07)<<13) | ((((long)(b>>3)) & 0x01F)<<8) | ((((long)(r>>3)) & 0x01F)<<3) | (((long)(g>>5)) & 0x07));
			
	//		ov_long = ov_long<<16;
			// do it for the next pixel too, so we have an int32 to put in the overlay (faster)
			rgbplane+=4;
			b1 = *(rgbplane);
			g1 = *(rgbplane+1);
			r1 = *(rgbplane+2);

			// original code
	//		ov_long = (ov_long & 0x0FFFF0000) | (((((long)(g1>>2)) & 0x07)<<13) | ((((long)(b1>>3)) & 0x01F)<<8) | ((((long)(r1>>3)) & 0x01F)<<3) | (((long)(g1>>5)) & 0x07));
			
			// thanks to BGA
			ov_long = ((r << 8) & (0x1f << 11)) + ((g << 3) & (0x3f << 5)) + ((b >> 3) & 0x1f) + ((r1 << 24) & (0x1f << 27)) + ((g1 << 19) & (0x3f << 21)) + ((b1 << 13) & (0x1f << 16));
			
			*(drawBits) = ov_long;
			if (paddingIndex++ == width) {
				drawBits+=margin;
				paddingIndex=1;
			}
		}
		
		renderer->drawBitmap->UnlockBits();
		renderer->decodeBitmap->UnlockBits();
		release_sem(renderer->decodeSemaphore);
	}
	return B_OK;
}

int32 MagicOverlayRenderer::DrawerThreadYCbCr422(void *arg)
{
	MagicOverlayRenderer *renderer;
	MediaView *view;
	
	renderer = (MagicOverlayRenderer *)arg;
	view = renderer->view;

	int32		*lt_r_y = renderer->lt_r_y;
	int32		*lt_g_y = renderer->lt_g_y;
	int32		*lt_b_y = renderer->lt_b_y;
	int32		*lt_r_cb = renderer->lt_r_cb;
	int32		*lt_g_cb = renderer->lt_g_cb;
	int32		*lt_b_cb = renderer->lt_b_cb;
	int32		*lt_r_cr = lt_b_cb; // same as lt_b_cb
	int32		*lt_g_cr = renderer->lt_g_cr;
	int32		*lt_b_cr = renderer->lt_b_cr;

//	register long cnt;
	register int r, g, b;
	register uchar y0, y1, cb, cr;
	
	register int32 ov_long;
	register int32 rgb_long;
	
	while(acquire_sem(renderer->drawSemaphore) == B_OK) {
		if (!renderer->drawBitmap || !view) break;
//		puts("MOR - got lock");
			
		renderer->decodeBitmap->LockBits();
		renderer->drawBitmap->LockBits();
		
		int32 * decodeBits = (int32*)(renderer->decodeBitmap->Bits());
		int32 * decodeEnd = (int32*)(((char*)decodeBits)+renderer->decodeBitmap->BitsLength());
		int32 * drawBits = (int32*)(renderer->drawBitmap->Bits());
		int32 * drawEnd = (int32*)(((char*)drawBits)+renderer->drawBitmap->BitsLength());
		int32 drawBytes = renderer->drawBitmap->BytesPerRow();
		int32 decodeBytes = renderer->decodeBitmap->BytesPerRow()/2; // colorspace conversion?
		
		int32 margin=(drawBytes-decodeBytes)/4;
		int32 width=decodeBytes/4;
		
		int32 paddingIndex = 1;
//		drawBits+=margin/2;

		uchar * rgbplane = (uchar*)decodeBits;
		uchar * rgb_end = (uchar*)decodeEnd;
		for( ; rgbplane<rgb_end && drawBits<drawEnd ; rgbplane+=4, drawBits++) {
			b = *(rgbplane);
			g = *(rgbplane+1);
			r = *(rgbplane+2);

			y0 = (uchar)((lt_r_y[r] + lt_g_y[g] + lt_b_y[b]) >> SCALEBITS);
			
#ifndef RGBYCC_NO_MEAN
			cb = (uchar)((lt_r_cb[r] + lt_g_cb[g] + lt_b_cb[b]) >> (SCALEBITS+1));
			cr = (uchar)((lt_r_cr[r] + lt_g_cr[g] + lt_b_cr[b]) >> (SCALEBITS+1));
#else
			cb = (uchar)((lt_r_cb[r] + lt_g_cb[g] + lt_b_cb[b]) >> (SCALEBITS));
			cr = (uchar)((lt_r_cr[r] + lt_g_cr[g] + lt_b_cr[b]) >> (SCALEBITS));
#endif
			rgbplane+=4;
			b = *(rgbplane);
			g = *(rgbplane+1);
			r = *(rgbplane+2);

#ifndef RGBYCC_NO_MEAN
			cb += (uchar)((lt_r_cb[r] + lt_g_cb[g] + lt_b_cb[b]) >> (SCALEBITS+1));
			cr += (uchar)((lt_r_cr[r] + lt_g_cr[g] + lt_b_cr[b]) >> (SCALEBITS+1));
#endif
			y1 = (uchar)((lt_r_y[r] + lt_g_y[g] + lt_b_y[b]) >> SCALEBITS);

			ov_long = ((y0&0x0FF) | ((cb<<8)&0x0FF00) | ((y1<<16)&0x0FF0000) | ((cr<<24)&0x0FF000000));
			
			// seems writing longs is faster than byte/byte
			// (dixit stampTV source)
			*(drawBits) = ov_long;
			if (paddingIndex++ == width) {
				drawBits+=margin;
				paddingIndex=1;
			}
		}
		
		renderer->drawBitmap->UnlockBits();
		renderer->decodeBitmap->UnlockBits();
//		puts("done rgb2ycc");
		release_sem(renderer->decodeSemaphore);
	}
	return B_OK;
}

