#include "MediaView.h"
#include "MediaSlider.h"
#include "TransportButton.h"
#include "Bitmaps.h"
#include "draw_window.h"
#include "messages.h"
#include "MediaFile.h"
#include "MediaTrack.h"
#include "AudioOutput.h"
#include "VideoOutput.h"
#include "SystemTime.h"
#include "colorspaces.h"
#include "BMTrack.h"

#include "VideoRenderer.h"
#include "DoubleBufferOverlayRenderer.h"
#include "DoubleBufferRenderer.h"
#include "DrawBitmapRenderer.h"
#include "DirectWindowRenderer.h"
#include "OverlayRenderer.h"
#include "MagicOverlayRenderer.h"

#include "main.h"

// debug
#include <SoundPlayer.h>

#include <Autolock.h>
#include <Bitmap.h>
#include <ScrollBar.h>
#include <Screen.h>
#include <Path.h>
#include <malloc.h>
#include <stdio.h>
#include <string.h>
#include <InterfaceDefs.h>
#include <WindowScreen.h>
#include <AppDefs.h>

#include "config.h"
#ifdef USE_SUBVIEW
#include "PictureView.h"
#endif

// DEBUG: BDataIO test
#include <File.h>

class MVLooper : public BLooper
{
public:
	MVLooper(MediaView *v, const char *name) : BLooper(name) {mv = v;};
	virtual ~MVLooper() {};
	virtual void DispatchMessage(BMessage *msg, BHandler *tgt);
private:
	MediaView *mv;
};


void MVLooper::DispatchMessage(BMessage *msg, BHandler *tgt)
{
	bigtime_t time, t2;
	int i;
	TrackPlayer *p;
	status_t err;
	bool playing;
	static bool scrub_playing; // was playing before scrubbing
	float vol = 0.5;
	
	switch(msg->what) {
		case MSG_OPEN:
			{
				bool needsZoom = !mv->HasVideoTrack();
				const char *p = NULL;
				BWindow *win = mv->Window();
				msg->PrintToStream();
				msg->FindString("file", &p);
				if(p==NULL)
					break;
				mv->Stop();

				if (win != 0) {
					BPath path(p);
					if (path.InitCheck() == B_OK) {
						if(win->Lock()) {
							win->SetTitle(strdup(path.Leaf()));
							win->Unlock();
						}
					}
				}

				mv->SetMediaSource(strdup(p));
				mv->Stop();
				
				if (win != 0) {
					float w = 0.0;
					float h = 0.0;
					if(win->Lock()) {
						mv->GetPreferredSize(&w, &h);	
						win->SetSizeLimits(80.0, 30000.0, (mv->HasVideoTrack()) ? 80.0 : h, (mv->HasVideoTrack()) ? 30000.0 : h);
		//				win->ResizeTo(w, h);
						win->Unlock();
					}
		//			mv->MakeFocus(true);
					
					needsZoom |= !mv->HasVideoTrack();
					
					if (needsZoom) 
						win->PostMessage(MSG_SETZOOM_1);
				}
				mv->Play();
			}
			break;
		case MSG_STOP:
			mv->Stop();
			time = 0;
			mv->SeekToTime(&time);
			break;
		case MSG_PLAY:
			if(mv->IsPlaying())
				break;
			mv->Play();
			break;
		case MSG_PAUSE:
			if(mv->IsPlaying())
				mv->Stop();
			else
				mv->Play();
			break;
		case MSG_SET_VOL:
			msg->FindFloat("to", &vol);
			if(mv->fAudioOutput && mv->fAudioOutput->player)
				mv->fAudioOutput->player->SetVolume(vol);
			break;
		case MSG_VOL_UP:
/*			if(!mv->fAudioOutput || !mv->fAudioOutput->player)
				break;
			vol = mv->fAudioOutput->player->Volume();
			vol+=0.1;
			vol = (vol>1.0)?(1.0):(vol);
			mv->fAudioOutput->player->SetVolume(vol);*/
			if(!mv->fMediaBar || !mv->fMediaBar->fVolButton)
				break;
			mv->fMediaBar->fVolButton->VolumeUp();
			break;
		case MSG_VOL_DOWN:
/*			if(!mv->fAudioOutput || !mv->fAudioOutput->player)
				break;
			vol = mv->fAudioOutput->player->Volume();
			vol-=0.1;
			vol = (vol<0.0)?(0.0):(vol);
			mv->fAudioOutput->player->SetVolume(vol);
*/			
			if(!mv->fMediaBar || !mv->fMediaBar->fVolButton)
				break;
			mv->fMediaBar->fVolButton->VolumeDown();
			break;
		case MSG_SEEK:
			if((playing=mv->IsPlaying()))
				mv->Stop();
			if(msg->FindInt64("to", &time)==B_NAME_NOT_FOUND) { // to absolute time
				time = mv->CurrentTime();
				if(msg->FindInt64("by", &t2)==B_NAME_NOT_FOUND)
					t2 = SEEK_AMOUNT; // by n microsec from CurrentTime()
				time += t2;
			}
			mv->SeekToTime(&time);
			if(playing)
				mv->Play();
			break;
		case MSG_SCRUB_BEG:
			mv->fScrubbing = true;
			if((scrub_playing=mv->IsPlaying()))
				mv->Stop();
			break;
		case MSG_SCRUB:
			if(!mv->fScrubbing)
				break;
			time = mv->fScrubTime;
			//mv->SeekToTime(&time); //don't seek audio (divx audio seek..)
			if(mv->fVideoOutput) {
				mv->fVideoOutput->SeekToTime(&time);
			}
//				mv->fMediaBar->SetCurTime(mv->CurrentTime());
//				snooze(100000);
			break;
		case MSG_SCRUB_END:
			time = mv->fScrubTime;
			mv->SeekToTime(&time);
			if(scrub_playing)
				mv->Play();
			mv->fScrubbing = false;
			break;
		case MSG_USE_RENDERER:
			{
				status_t err;
				VideoRenderer *p = NULL;
				err = msg->FindPointer("renderer", (void **)&p);
				if(p==NULL)
					break;
				err = mv->fVideoOutput->TryRenderer(p);
				if(err != B_OK)
					err = mv->fVideoOutput->FindRenderer();
			}
			break;
		case MSG_USE_TIMEREF:
			{
				if((playing=mv->IsPlaying()))
					mv->Stop();

				status_t err;
				TimeSource *p = NULL;
				err = msg->FindPointer("timeref", (void **)&p);
				if(p==NULL)
					break;
				time = mv->time_source->CurrentTime();
				p->SeekToTime(&time);
//				if(playing)
//					p->Play();
//				else
					p->Stop();
				printf("using timeref: %08lX\n", p);
				time = mv->time_source->CurrentTime();

				mv->time_source = p;
				if(playing)
					mv->Play();

			}
			break;
		case MSG_EN_HACK_DIVX_SEEK:
			settings->enable_divx_seek_hack = !settings->enable_divx_seek_hack;
			break;
		case MSG_EN_HACK_DIVX_SYNC:
			settings->enable_divx_sync_hack = !settings->enable_divx_sync_hack;
			break;
		default:
		BLooper::DispatchMessage(msg,tgt);
		break;
	}
}



MediaView::MediaView(
	BRect		frame, 
	const char	*name,
	uint32		resizeMask,
	uint32		flags)
		: BView(frame, name, resizeMask, flags)
{
	InitObject();
	looper = (BLooper *)new MVLooper(this, name);
	looper->Run();
}


MediaView::~MediaView()
{
	Stop();
	Reset();
	if(fRenderers) {
		for (int i = fRenderers->CountItems () - 1; i >= 0; i--) {
			delete fRenderers->ItemAtFast (i);
			fRenderers->RemoveItem(i);
		}
//		delete fRenderers;
// FIWME: WHY DOES IT CRASH ???
	}
#ifdef USE_SUBVIEW
	picture = NULL;
#endif
	looper->Lock(); // exit the BLooper
	looper->Quit();
}

void MediaView::MessageReceived(BMessage *message)
{
	entry_ref ref;
	int32	count = 0;
	uint32	type = 0;

	switch(message->what) {
	case B_SIMPLE_DATA: // drag'n'drop from the tracker, just repackage the path
		message->GetInfo("refs", &type, &count);
		if(count<1)
			break;
		if (message->FindRef("refs", 0, &ref) == B_NO_ERROR) {
			BEntry entry(&ref);

			if (entry.InitCheck() == B_NO_ERROR) {
				BPath path;
				entry.GetPath(&path);
				BMessage *msg;
				msg = new BMessage(MSG_OPEN);
				msg->AddString("file", strdup(path.Path()));
				looper->PostMessage(msg);
			}
		}

		if(count>1 && count<10) { // open the others in new windows, only if <10 :-)
			message->RemoveData("refs", 0); // remove the first one, already opened 
			message->what = B_REFS_RECEIVED;
			be_app->PostMessage(message);
		}
		message->PrintToStream();
		break;
	default:
		message->PrintToStream();
		BView::MessageReceived(message);
		break;
	}
}

void MediaView::KeyDown(const char *bytes, int32 numBytes)
{
	float vol;
	BMessage *msg;
	
	switch(bytes[0])
	{
/*		case B_SPACE:
			Pause();
			break;*/ // done already
		case B_UP_ARROW:
			looper->PostMessage(MSG_VOL_UP);
			break;
		case B_DOWN_ARROW:
			looper->PostMessage(MSG_VOL_DOWN);
			break;
		case B_RIGHT_ARROW:
			looper->PostMessage(MSG_SEEK);
			break;
		case B_LEFT_ARROW:
			{
				BMessage *msg = new BMessage(MSG_SEEK);
				msg->AddInt64("by", -(SEEK_AMOUNT));
				looper->PostMessage(msg);
			}
			break;
		// topmost
		case 't':
		case 'T':
			Window()->SetFeel((window_feel)((Window()->Feel() & (~B_FLOATING_ALL_WINDOW_FEEL)) | (Window()->Feel() & (B_FLOATING_ALL_WINDOW_FEEL))?(0):(B_FLOATING_ALL_WINDOW_FEEL)));
			break;
		// fullscreen
		case B_ESCAPE:
		case 'f':
		case 'F':
			Window()->Zoom();
//				((PlayerWindow)(Window()))->SetFullScreen(!((PlayerWindow)(Window()))->FullScreen());

			break;
		// as Desktop
		case 'd':
		case 'D':
//			Window()->Set()
			break;
		// CL-Amp like controls
		case 'x':
		case 'X':
			looper->PostMessage(MSG_PLAY);
			break;
		case 'c':
		case 'C':
//			Control(MEDIA_PLAY);
			looper->PostMessage(MSG_PAUSE);
			break;
		case 'v':
		case 'V':
//			Control(MEDIA_STOP);
			looper->PostMessage(MSG_STOP);
			break;
		default:
			BView::KeyDown(bytes, numBytes);
			break;
	}
}

void MediaView::MouseDown(BPoint p)
{
	BWindow *window = Window();	/*To handle the MouseDown message*/
	if (!window)	/*Check for proper instantiation*/
		return;
		
	BMessage *mouseMsg = window->CurrentMessage();
	if (!mouseMsg)	/*Check for existence*/
		return;
		

	if (mouseMsg->what == B_MOUSE_DOWN) {
		mouseMsg->what = MSG_POPUP;
		Window()->PostMessage(mouseMsg);
//		PostMessage(mouseMsg);
	}// else
//		BView::MouseDown(p);

}

float MediaView::Volume()
{
	if(!fAudioOutput || !fAudioOutput->player)
		return 0;
	return fAudioOutput->player->Volume();
	
}

void MediaView::Pulse()
{
	if (fFullScreen && idle_time() > 29000000) {
		BPoint where;
		uint32 buttons;
		GetMouse(&where, &buttons, false);
		ConvertToScreen(&where);
		set_mouse_position((int32) where.x, (int32) where.y);
	}
	if(fScrubbing)
		fMediaBar->SetCurTime(fScrubTime);
	else
		fMediaBar->SetCurTime(CurrentTime());
	if(!fScrubbing)
		if(IsPlaying())
			fMediaBar->ActionUpdate(MEDIA_PLAY);
		else
			fMediaBar->ActionUpdate(MEDIA_STOP);
}


MediaTrack		*MediaView::VideoTrack()
{
	return fVideoTrack;
}

MediaTrack		*MediaView::AudioTrack()
{
	return fAudioTrack;
}

status_t MediaView::SetMediaSource(const char *path)
{
	BAutolock autolock(Window());
	status_t	err = B_ERROR;
	entry_ref	ref;
	
	float		v_rate, a_rate;
	
	Reset(); // we want to reuse the same window.... no for now, but when playlist comes...
	
	err = get_ref_for_path(path, &ref);
	if (err != B_NO_ERROR)
		return (err);

	time_source = sys_time = new SystemTime;

	fMediaFile = new BMediaFile(&ref);

//    BView * view = fMediaFile->GetParameterView();
//    if (view != 0) {
//      float width, height;
//      view->GetPreferredSize(&width,&height);
//      BWindow * window = new BWindow(BRect(50,50,100,100),"wow",B_FLOATING_WINDOW,0);
//      window->AddChild(view);
//      window->Show();
//    }

	// BDataIO test
//	fMediaFile = new BMediaFile(new BFile(&ref, B_READ_ONLY), B_MEDIA_FILE_NO_READ_AHEAD);
	
	bool	foundTrack = false;
	int32	numTracks = fMediaFile->CountTracks();

	for (int32 i = 0; i < numTracks; i++) {
		BMediaTrack *btrack = fMediaFile->TrackAt(i);
		BMTrack *track;
		
		if (btrack == NULL) {
			Reset();
			return (B_ERROR);
		}
		else {
			bool			trackUsed = false;
			media_format	mf;

			if (btrack->EncodedFormat(&mf) == B_NO_ERROR) {			
				switch (mf.type) {
					case B_MEDIA_ENCODED_VIDEO:
printf("#################field rate %f\n", mf.u.encoded_video.output.field_rate);
						track = new BMTrack(btrack);
						trackUsed = SetVideoTrack(path, track, &mf) == B_NO_ERROR;
						if(trackUsed)
							v_rate = mf.u.encoded_video.output.field_rate;
						break;
	
					case B_MEDIA_RAW_VIDEO:
printf("#################field rate %f\n", mf.u.raw_video.field_rate);
						track = new BMTrack(btrack);
						trackUsed = SetVideoTrack(path, track, &mf) == B_NO_ERROR;
						if(trackUsed)
							v_rate = mf.u.raw_video.field_rate;
						break;
	
					case B_MEDIA_RAW_AUDIO:
						track = new BMTrack(btrack);
						trackUsed = SetAudioTrack(path, track, &mf) == B_NO_ERROR;
						if(trackUsed)
							a_rate = mf.u.raw_audio.frame_rate;
						break;
						
					case B_MEDIA_ENCODED_AUDIO:
						if (btrack->DecodedFormat(&mf) == B_NO_ERROR) {
							track = new BMTrack(btrack);
							trackUsed = SetAudioTrack(path, track, &mf) == B_NO_ERROR;
						}
						if(trackUsed)
							a_rate = mf.u.encoded_audio.output.frame_rate;
						break;

					default:
						break;
				}
			}
	
			if (trackUsed)
				foundTrack = true;
			else {
				fMediaFile->ReleaseTrack(btrack);
			}
		}
	}

	if (foundTrack) {
		status_t err = B_ERROR;

		if(fAudioTrack != NULL)
			audio_frame = (bigtime_t)(((float)1000000LL)/a_rate);
		if(fVideoTrack != NULL)
			video_frame = (bigtime_t)(((float)1000000LL)/v_rate);
		
		if(fAudioOutput != NULL) {
//			delete time_source; // DON'T DO THAT !
			time_source = fAudioOutput;
		}
		
		if(!fAudioOutput)
			fMediaBar->fVolButton->SetEnabled(false);
		else {
			fMediaBar->fVolButton->SetEnabled(true);
			fMediaBar->fVolButton->SetVolume(0.75);
		}
		
			fMediaBar->SetTotalTime(Duration());
		
		thread_id extractor_reader = find_thread("extractor_reader");
		if (extractor_reader != B_NAME_NOT_FOUND) {
			set_thread_priority(extractor_reader,B_DISPLAY_PRIORITY);
		}
		
	}

	return (B_NO_ERROR);
}

// not to be used
status_t
MediaView::SetColorSpace(
	color_space	depth)
{
	BAutolock autolock(Window());

	fBitmapDepth = depth;

	return (B_NO_ERROR);
}


// not to be used
color_space
MediaView::ColorSpace() const
{
	return (fBitmapDepth);
}

VideoRenderer *MediaView::CurrentRenderer() const
{
	return fVideoOutput->RendererUsed();
}


status_t
MediaView::Control(
	media_action	action)
{
	BAutolock autolock(Window());

	status_t err = B_NO_ERROR;

	switch (action) {
		case MEDIA_PLAY:
		/*
			if(time_source != NULL)
				time_source->Stop();
			FOR_EACH_TRACKPLAYER // p is a pointer to the player
				if(p != NULL && p != time_source)
					err = p->Play();
//						if(err != B_OK)
//							break;
		*/
			Play();
			break;

		case MEDIA_STOP:
		/*
			if(time_source != NULL)
				time_source->Stop();
			FOR_EACH_TRACKPLAYER // p is a pointer to the player
				if(p != NULL && p != time_source)
					err = p->Stop();
//						if(err != B_OK)
//							break;
		*/
			Stop();
			break;

		default:
			err = B_ERROR;
			break;
	}

	fMediaBar->ActionUpdate(action);

	return (err);
}


bool
MediaView::IsPlaying() const
{
	return (fPlaying);
}


bool
MediaView::HasVideoTrack() const
{
	return (fVideoTrack != NULL);
}


bool
MediaView::HasAudioTrack() const
{
	return (fAudioTrack != NULL);
}


void
MediaView::GetPreferredSize(
	float	*width,
	float	*height)
{
	if (fVideoOutput == NULL) {
 		BView::GetPreferredSize(width, height);
		*height = kMediaBarHeight - 1.0;
	}
	else {
//		BRect bitmapBounds = fBitmap->Bounds();
		BRect bitmapBounds = fVideoOutput->SourceBounds();
//printf("bm = %f %f\nrect=%f %f %f %f\n", bitmapBounds.Width(), bitmapBounds.Height(), bitmapBounds.left, bitmapBounds.top ,bitmapBounds.right ,bitmapBounds.bottom);
		*width = bitmapBounds.Width();
		*height = bitmapBounds.Height() + ((fFullScreen)?(0):(kMediaBarHeight));
	}
}


void
MediaView::Draw(
	BRect	updateRect)
{
if(!IsPlaying() && fVideoOutput && fVideoOutput->RendererUsed())
	fVideoOutput->RendererUsed()->ReDraw();
}

void MediaView::SetFullScreen(bool fs)
{

	if(fs && !fFullScreen)
		fMediaBar->Hide();
	else if(!fs && fFullScreen)
		fMediaBar->Show();
	fFullScreen = fs;
}



void
MediaView::DetachedFromWindow()
{
	Stop(); // does the same thing as above
/*
	if(time_source != NULL)
		time_source->Stop();
	FOR_EACH_TRACKPLAYER
		if(p != NULL && p != time_source)
			p->Stop();
*/
}


void
MediaView::FrameResized(
	float	width,
	float	height)
{
	float movie_width;
	float movie_height;
	
	
#ifdef USE_SUBVIEW
	if(fVideoOutput) {
		float x,y, w,h;
		BScreen bs;
		height -= (fFullScreen?(0):(kMediaBarHeight));
		bool cinemascope; // true if the sides touch, false for top-bottom
		float xscale, yscale, scale;
		movie_width = fVideoOutput->SourceBounds().right - fVideoOutput->SourceBounds().left + 1;
		movie_height = fVideoOutput->SourceBounds().bottom - fVideoOutput->SourceBounds().top + 1;
		cinemascope = (xscale = (width/movie_width)) < (yscale = (height/movie_height));
		scale = cinemascope?xscale:yscale;
		x=(float)(long)(cinemascope?(0):((width-scale*movie_width)/2));
		y=(float)(long)(cinemascope?((height-scale*movie_height)/2):(0));
		w=(float)(long)(cinemascope?(width):(scale*movie_width));
		h=(float)(long)(cinemascope?(scale*movie_height):(height));
		
		if(fFullScreen) {
//			x = (float)(long)(((long)(x+7))/8)*8;;
//			y = (float)(long)(((long)(y+7))/8)*8;;
			x = (float)(long)(((long)(x+15))/16)*16;;
			y = (float)(long)(((long)(y+15))/16)*16;;
//			w=(float)(long)((long)(cinemascope?(width):(scale*movie_width))/4)*4+1;
//			h=(float)(long)((long)(cinemascope?(scale*movie_height):(height))/4)*4+1;
//			w = (float)(long)(((long)(w+7))/8)*8;
//			h = (float)(long)(((long)(h+7))/8)*8;
//			h=bs.Frame().bottom+1;
			w = (float)(long)(((long)(w+15))/16)*16;
			h = (float)(long)(((long)(h+15))/16)*16;
//			picture->MoveTo((float)(long)(((long)(cinemascope?(0):((width-scale*movie_width)/2))/4)*4), (float)(long)(((long)(cinemascope?((height-scale*movie_height)/2):(0))/4)*4));
//			picture->MoveTo(0,0);
		}
//x++; y++;
		w--; // hmmm... size ?
		h--;
		printf("ResizeTo(%f, %f);\n", w, h);
		picture->ResizeTo(w, h);
		printf("MoveTo(%f, %f);\n", x, y);
		picture->MoveTo(x, y);
		picture->ResizeTo(w, h);
		picture->MoveTo(x, y);
	} else {
		picture->ResizeTo(0,0);
	}
#endif
	
	
	Draw(Bounds());
}


void
MediaView::InitObject()
{
	BScreen screen;
	rgb_color vcol= B_TRANSPARENT_32_BIT;

	fMediaFile = NULL;
	fVideoTrack = NULL;
	fAudioTrack = NULL;
	fVideoOutput = NULL;
	fAudioOutput = NULL;
	fMediaBar = NULL;
	fBitmap = NULL;
	fBitmapDepth = screen.ColorSpace();
	fCurTime = 0;
	fPlayerThread = B_ERROR;
	fPlaySem = B_ERROR;
	fScrubSem = B_ERROR;
	fPlaying = false;
	fSnoozing = false;
	fAudioDumpingBuffer = NULL;
	videoOffsetTime = 0LL;
	audioOffsetTime = 0LL;
	fFullScreen = false;
	sys_time = NULL;
	time_source = NULL;

	endOfVideoTrack=false;

#ifdef USE_SUBVIEW
	picture = new PictureView(this, Bounds(), "picture", B_FOLLOW_NONE/*B_FOLLOW_ALL_SIDES*/, B_WILL_DRAW);
	AddChild(picture);
#endif
	BRect mediaBarFrame = Bounds();
	mediaBarFrame.top = mediaBarFrame.bottom - kMediaBarHeight;
	fMediaBar = new _MediaBar_(mediaBarFrame, this);
	AddChild(fMediaBar);

#ifdef USE_SUBVIEW
	picture->SetViewColor(B_TRANSPARENT_32_BIT);
	vcol.red = 0;
	vcol.green = 0;
	vcol.blue = 0;
	vcol.alpha = 0;
#endif
	SetViewColor(vcol);
	
	
	fRenderers = new TList<VideoRenderer>;
//BGA
#ifdef HACK_BGA // ;-)
	fRenderers->AddItem(new MagicOverlayRenderer);
	fRenderers->AddItem(new OverlayRenderer);
#else
	fRenderers->AddItem(new OverlayRenderer);
	fRenderers->AddItem(new DoubleBufferOverlayRenderer);
	fRenderers->AddItem(new MagicOverlayRenderer);
#endif
	fRenderers->AddItem(new DoubleBufferRenderer);
	fRenderers->AddItem(new DrawBitmapRenderer);
	fRenderers->AddItem(new DirectWindowRenderer); // experimental
}


status_t
MediaView::SetVideoTrack(
	const char      *path,
	MediaTrack		*track,
	media_format	*format)
{
	if (fVideoTrack != NULL)
		// is it possible to have multiple video tracks?
		return (B_ERROR);

	fVideoTrack = track;
	
	fVideoOutput = new VideoOutput(fVideoTrack, strdup(BPath(path).Leaf()), format, this);
	status_t err = fVideoOutput->InitCheck();
	if (err != B_NO_ERROR) {
		delete (fVideoOutput);
		fVideoOutput = NULL;
		fVideoTrack = NULL;

		return (err);
	}
	return (B_NO_ERROR);
}


status_t
MediaView::SetAudioTrack(
	const char      *path,
	MediaTrack		*track,
	media_format	*format)
{
	if (fAudioTrack != NULL)
		// is it possible to have multiple tracks?
		return (B_ERROR);

	fAudioTrack = track;

	fAudioOutput = new AudioOutput(this, fAudioTrack, strdup(BPath(path).Leaf()));
	status_t err = fAudioOutput->InitCheck();
	if (err != B_NO_ERROR) {
		delete (fAudioOutput);
		fAudioOutput = NULL;
		fAudioTrack = NULL;

		return (err);
	}

	fAudioDumpingBuffer = malloc(format->u.raw_audio.buffer_size);

	return (B_NO_ERROR);
}


status_t
MediaView::Play()
{
	status_t err;
	int i;
	TrackPlayer *p;
	
	if (fPlaying)
		return (B_NO_ERROR);

	fPlaying = true;
// ?
	release_sem(fPlaySem);

	if(time_source != NULL)
		time_source->Play();
	FOR_EACH_TRACKPLAYER(i, p) // p is a pointer to the player
	{
			printf("##starting obj %08lX\n", p);
		
		if(p != NULL && p != time_source) {
			printf("starting obj %08lX\n", p);
			err = p->Play();
		}
	}

	return (B_NO_ERROR);
}


status_t
MediaView::Stop()
{
	int32 c;
	status_t err;
	int i;
	TrackPlayer *p;
	
	if (!fPlaying)
		return (B_NO_ERROR);
	
	acquire_sem(fPlaySem);
	fPlaying = false;
	
	if(time_source != NULL)
		time_source->Stop();
	FOR_EACH_TRACKPLAYER(i, p) // p is a pointer to the player
		if(p != NULL && p != time_source)
			err = p->Stop();

//
//	while(B_OK==get_sem_count(fPlaySem, &c) && c>=0) snooze(5000);

	return (B_NO_ERROR);
}


void
MediaView::Reset()
{
/*
	delete_sem(fPlaySem);
	fPlaySem = B_ERROR;
*/

// what to do with that ?
	if(fScrubSem!=B_ERROR)
		delete_sem(fScrubSem);
	fScrubSem = B_ERROR;

	status_t result = B_NO_ERROR;

	if(fVideoOutput != NULL) {
		fVideoOutput->Stop();
		delete (fVideoOutput);
	}
	fVideoOutput = NULL;
	
	if(fAudioOutput != NULL)
		delete (fAudioOutput);
	fAudioOutput = NULL;

	if(fVideoTrack != NULL)
		delete fVideoTrack;
	fVideoTrack = NULL;

	if(fAudioTrack != NULL)
		delete fAudioTrack;
	fAudioTrack = NULL;

	time_source = NULL;

	if(sys_time)
		delete sys_time;
	sys_time = NULL;


	if(fMediaFile != NULL) {
		delete (fMediaFile);
	}
	fMediaFile = NULL;
	
//	free(fAudioDumpingBuffer);
	fAudioDumpingBuffer = NULL;
}


BRect
MediaView::VideoBounds() const
{
//puts("MediaView::VideoBounds()");
#ifdef USE_SUBVIEW
	BRect videoBounds = picture->Bounds();
//	videoBounds.bottom -= kMediaBarHeight + 1;
#else
	BRect videoBounds = Bounds();
	videoBounds.bottom -= kMediaBarHeight;
#endif
	return (videoBounds);
}

bigtime_t MediaView::CurrentTime()
{
	if(time_source == NULL)
		return 0LL;
	return ((TimeSource *)time_source)->CurrentTime();
}

bigtime_t MediaView::Duration()
{
	bigtime_t video, audio;
	video = 0;
	audio = 0;
	if(fAudioTrack)
		audio = fAudioTrack->Duration();
	if(fVideoTrack)
		video = fVideoTrack->Duration();
//	return (audio<video)?(video):(audio);
	return (fVideoTrack)?(video):(audio);
}

bool MediaView::EndOfVideoTrack()
{
	return (fVideoTrack)?(fVideoOutput->EndOfTrack()):(false);
}

bool MediaView::EndOfAudioTrack()
{
	return (fAudioOutput)?(fAudioOutput->EndOfTrack()):(false);
}

void MediaView::NotifyEndOfTrack(TrackPlayer *which)
{
/*
	bigtime_t time;
	Stop();
	time = 0;
	SeekToTime(&time);
*/
looper->PostMessage(MSG_STOP);
}

status_t MediaView::SeekToTime(bigtime_t *inout_time)
{
	bigtime_t time;
	int i;
	TrackPlayer *p;
	status_t err;
	
	Stop();
	time = *inout_time;
	if(time_source != NULL)
		time_source->SeekToTime(&time);
	*inout_time = time;
	FOR_EACH_TRACKPLAYER(i, p) // p is a pointer to the player
		if(p != NULL && p != time_source)
			err = p->SeekToTime(&time);
	return B_OK;
}


