/* wmouse.cpp
 *
 * Let's see if we can access the serial port!
 */

//#define DEBUG 1
#include <support/Debug.h>
#include "wmouse.h"

//void do_something_with_it(unsigned char);
int main(int, char *[]);

bool WMInputDevice::wmActive = false;
thread_id WMInputDevice::wm_thread = B_ERROR;
struct current_state *WMInputDevice::state = NULL;
BSerialPort *WMInputDevice::serial = NULL;
struct pnp_info *WMInputDevice::pnpinfo = NULL;
bigtime_t WMInputDevice::click_speed = 100000L;
int32 WMInputDevice::mouse_speed = 0;
mouse_map *WMInputDevice::current_mouse_map = NULL;
int32 *WMInputDevice::accelerated = NULL;
bool WMInputDevice::i_should_send_double_clicks_for_fmb_presses = true;

BInputServerDevice*
instantiate_input_device()
{
	return (new WMInputDevice());
}

WMInputDevice::WMInputDevice() : BInputServerDevice()
{
	input_device_ref	wmDevice				= { "Wheely mouse",
													B_POINTING_DEVICE,
													NULL };
	input_device_ref	*wmDeviceList[2]		= {&wmDevice, NULL};

    SERIAL_PRINT(("wmouse: in constructor\n"));

    state = (struct current_state *)malloc(sizeof(struct current_state));
    if(state) {
      state->dx = 0L;
      state->dy = 0L;
      state->buttons_old = 0L;
      state->buttons_new = 0L;
      state->wheel_change = 0L;
    }

    current_mouse_map = (mouse_map *)malloc(sizeof(mouse_map));
    accelerated = (int32 *)malloc(256 * sizeof(int32));

    serial = new BSerialPort;

	RegisterDevices(wmDeviceList);	
}

WMInputDevice::~WMInputDevice()
{
  // clean up
  if(pnpinfo) free(pnpinfo);
  delete serial;
  if(state) free(state);
  if(current_mouse_map) free(current_mouse_map);
  if(accelerated) free(accelerated);
  SERIAL_PRINT(("wmouse: in destructor\n"));
}

status_t
WMInputDevice::InitCheck()
{
    SERIAL_PRINT(("wmouse: InitCheck()\n"));
    if(!(state)) return(false);
    if(!(current_mouse_map)) return(false);
    if(!(accelerated)) return(false);
    else accelerated += 128;
    if(serial == NULL) return(false);

	return (BInputServerDevice::InitCheck());
}

status_t
WMInputDevice::SystemShuttingDown()
{
    SERIAL_PRINT(("wmouse: SystemShuttingDown()\n"));
	return (BInputServerDevice::SystemShuttingDown());
}

status_t
WMInputDevice::Start(const char *device, void *cookie)
{
  wmActive = true;
  wm_thread = spawn_thread(do_it_serial, device, B_DISPLAY_PRIORITY, this);
  
  resume_thread(wm_thread);
  return(B_NO_ERROR);
}

status_t
WMInputDevice::Stop(const char *device, void *cookie)
{
  status_t err = B_OK;
  
  wmActive = false;
  wait_for_thread(wm_thread, &err);
  wm_thread = B_ERROR;
  return (B_NO_ERROR);
}

status_t
WMInputDevice::Control(const char *device, void *cookie, uint32 command, BMessage *message)
{
SERIAL_PRINT(("wmouse: Control()\n"));
  switch(command) {
    case B_CLICK_SPEED_CHANGED:
      get_click_speed(&click_speed);
      break;
    case B_MOUSE_MAP_CHANGED:
      get_mouse_map(current_mouse_map);
      break;
    case B_MOUSE_SPEED_CHANGED:
      get_mouse_speed(&mouse_speed);
      SERIAL_PRINT(("wmouse: speed now %ld\n", mouse_speed));
      calculate_acceleration_table(mouse_speed);
      break;
    case B_MOUSE_TYPE_CHANGED:
      SERIAL_PRINT(("wmouse: B_MOUSE_TYPE_CHANGED not implemented\n"));
      break;
    default:
      SERIAL_PRINT(("wmouse: unknown Control() message received\n"));
      break;
  }
  
  return(B_OK);
}

int32 WMInputDevice::do_it_serial(void *arg)
{
  char devName[B_OS_NAME_LENGTH];
  unsigned char buffer = 0;
  unsigned char buf4[4];
  int bytes_read = 0;

  for (int32 n = serial->CountDevices()-1; n >= 0; n--) {
    serial->GetDeviceName(n, devName);
    if ( serial->Open(devName) > 0 ) {
      SERIAL_PRINT(("wmouse: device no. %ld: `%s' opened\n", n, devName));
      serial->SetDataRate(B_1200_BPS);
      serial->SetDataBits(B_DATA_BITS_7);
      serial->SetStopBits(B_STOP_BITS_2);
      // identification
      serial->Read((void *)&buffer, 1);
      if (!(buffer == 'M')) {
        SERIAL_PRINT(("wmouse: read 0x%02x; doesn't look like a mouse on %s to me---closing\n", buffer, devName));
        serial->Close();
        continue;  // this *does* do `NEXT n', doesn't it?
      }

      // at present, we assume that 4-byte protocol
      // later on, we will try detecting the device attached through Plug 'n' Pray
      // (if available (otherwise we will assume M$ (or perhaps Mouse systems)))

      // initial synchronisation
      detect_and_synchronise();
      get_mouse_map(current_mouse_map);
      get_click_speed(&click_speed);
      get_mouse_speed(&mouse_speed);
      calculate_acceleration_table(mouse_speed);
      
SERIAL_PRINT(("wmouse: about to attempt to read from the serial port\n"));
      while(wmActive) {
          if ((bytes_read = serial->Read((void *) buf4, 4)))
            process_bytes(arg, buf4, bytes_read);
      }
      serial->Close();
    }
    else { SERIAL_PRINT(("wmouse: device no. %ld: `%s' could not be opened\n", n, devName)); }
  }
  return 0;
}

void WMInputDevice::notify_changes(void *arg)
{
  bigtime_t	now = system_time();
  BMessage	*event;
  WMInputDevice *wmDevice = (WMInputDevice *)arg;
  int32 buttons_change = (state->buttons_new ^ state->buttons_old);
  int bit = 0;

  SERIAL_PRINT(("nc(): dx = %d, dy = %d, bo = %x, bn = %x, w = %d\n",
    state->dx, state->dy, state->buttons_old, state->buttons_new, state->wheel_change));

  // buttons changed
  // for each bit in buttons_change, send button down or up as appropriate
  while (buttons_change) {
    if (buttons_change & 0x01) {
      if (state->buttons_new & (1 << bit)) {
        // send_down(1 << bit);
        event = new BMessage(B_MOUSE_DOWN);
        event->AddInt64("when", now);
        event->AddInt32("x", 0L);
        event->AddInt32("y", 0L);
        event->AddInt32("buttons", state->buttons_new);
        if((state->last_button_clicked == (1 << bit))
           && (now - state->last_click_time < double_click_time())) {
          ++state->clicks;
        }
        else {
          state->clicks = 1;
        }
        event->AddInt32("clicks", state->clicks);
        wmDevice->EnqueueMessage(event);
SERIAL_PRINT(("wmouse: detected button %d down (clicks == %d)\n", bit, state->clicks));
        state->last_button_clicked = 1 << bit;
        state->last_click_time = now;
      }
      else {
        // send_up(1 << bit);
        event = new BMessage(B_MOUSE_UP);
        event->AddInt64("when", now);
        event->AddInt32("x", 0L);
        event->AddInt32("y", 0L);
        event->AddInt32("buttons", (state->buttons_new & LMB ? B_PRIMARY_MOUSE_BUTTON : 0)
								 | (state->buttons_new & RMB ? B_SECONDARY_MOUSE_BUTTON : 0)
								 | (state->buttons_new & MMB ? B_TERTIARY_MOUSE_BUTTON : 0)
								 | (state->buttons_new & FMB ? B_QUATERNARY_MOUSE_BUTTON : 0));
        event->AddInt32("clicks", state->clicks);
        wmDevice->EnqueueMessage(event);
SERIAL_PRINT(("wmouse: detected button %d up\n", bit));
      }
    }
    ++bit;
    buttons_change >>= 1;
  }

  // mouse moved
  if ((state->dx != 0) || (state->dy != 0) || state->wheel_change) {
    event = new BMessage(B_MOUSE_MOVED);
    event->AddInt64("when", now);
    event->AddInt32("buttons", (state->buttons_new & LMB ? B_PRIMARY_MOUSE_BUTTON : 0)
							 | (state->buttons_new & RMB ? B_SECONDARY_MOUSE_BUTTON : 0)
							 | (state->buttons_new & MMB ? B_TERTIARY_MOUSE_BUTTON : 0)
							 | (state->buttons_new & FMB ? B_QUATERNARY_MOUSE_BUTTON : 0));
    event->AddInt32("x", accelerated[state->dx]);
    event->AddInt32("y", -accelerated[state->dy]);
//    event->AddInt32("wheel_delta", state->wheel_change);
    event->AddInt32("clicks", state->clicks);
    
    wmDevice->EnqueueMessage(event);
  }
  
  // formerly subsumed by B_MOUSE_MOVED
  if (state->wheel_change) {
    SERIAL_PRINT(("wmouse: detected wheel change of %d\n", state->wheel_change));
    // send wheel message
    for (int i = 0; i < abs(state->wheel_change); ++i) {
      // Here's where the keys sent are decided.
      // Choices are in <interface/InterfaceDefs.h> and include
      // B_PAGE_UP, B_PAGE_DOWN, B_UP_ARROW, B_DOWN_ARROW,
      // B_LEFT_ARROW, and B_RIGHT_ARROW .  (Don't forget to change
      // send_key.cpp as well.)
      send_key(arg, (state->wheel_change) > 0 ? B_PAGE_DOWN : B_PAGE_UP);
    }
  }
  
  if(i_should_send_double_clicks_for_fmb_presses
      && (!(state->buttons_old & B_QUATERNARY_MOUSE_BUTTON))
      && (state->buttons_new & B_QUATERNARY_MOUSE_BUTTON)) {
    // send a double-click
    SERIAL_PRINT(("wmouse: about to send a double-click\n"));
    for (int32 j = 0; j < 3; ++j) {
      BMessage *dbl_event = new BMessage((j&1) ? B_MOUSE_UP : B_MOUSE_DOWN);
      dbl_event->AddInt64("when", system_time());
      dbl_event->AddInt32("buttons", (j&1) ? (state->buttons_new & (!B_PRIMARY_MOUSE_BUTTON))
                                           : (state->buttons_new | B_PRIMARY_MOUSE_BUTTON));
      dbl_event->AddInt32("x", 0L);
      dbl_event->AddInt32("y", 0L);
      dbl_event->AddInt32("wheel_delta", 0L);
      if (!(j&1)) dbl_event->AddInt32("clicks", (j/2) + 1);
      wmDevice->EnqueueMessage(dbl_event);
    }
  }
  if(i_should_send_double_clicks_for_fmb_presses
     && (state->buttons_old & B_QUATERNARY_MOUSE_BUTTON)
     &&(!(state->buttons_new & B_QUATERNARY_MOUSE_BUTTON))) {
    // send LMB up
    BMessage *up_event = new BMessage(B_MOUSE_UP);
    up_event->AddInt64("when", system_time());
    up_event->AddInt32("buttons", state->buttons_new & (!B_PRIMARY_MOUSE_BUTTON));
    up_event->AddInt32("x", 0L);
    up_event->AddInt32("y", 0L);
    up_event->AddInt32("wheel_delta", 0L);
    wmDevice->EnqueueMessage(up_event);
  }
}

inline bigtime_t WMInputDevice::double_click_time(void) {
  return click_speed;
}

void WMInputDevice::calculate_acceleration_table(int32 speed)
{
//  float k = exp(speed - 5.0);
  float k;

  // simple direct mapping
  if (speed == 1) {
    for (int i = -128; i <128; ++i) {
      accelerated[i] = i;
    }
  }

  k = (speed + 1)/2.0;
  
  if (k < 1.0) {
    for (int i = 0 ; i < 128 ; ++i) {
      accelerated[i] = (int32)(k * i);
      accelerated[-i] = -accelerated[i];
    }
    accelerated[-128] = (int32)(k * (-128));
  } else {  // k >= 1.0
    for (int i = 0 ; i < 128 ; ++i) {
      accelerated[i] = (int32)((k * i) - 2*(k-1)*(sqrt(i+1)-1));
      accelerated[-i] = -accelerated[i];
    }
    accelerated[-128] = (int32)((k*128) - 2*(k-1)*(sqrt(128+1)-1));
  }  // k < 1.0
}
