
//		Copyright © 1999		Matthew S. Chartier (mattc@ic.net)
//
//		This program is free software; you can redistribute it and/or modify
//		it under the terms of the GNU General Public License as published by
//		the Free Software Foundation; either version 1, or (at your option)
//		any later version.
//
//		This program is distributed in the hope that it will be useful,
//		but WITHOUT ANY WARRANTY; without even the implied warranty of
//		MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	See the
//		GNU General Public License for more details.
//
//		You should have received a copy of the GNU General Public License
//		along with this program; if not, write to the Free Software
//		Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//
//		Portions of this file Copyright David Cambell (cambell@torque.net)		
//										Tim Waugh (tim@cyberelk.demon.co.uk)
//										Jose Renau (renau@acm.org)


#ifdef __USERSPACE
#include <stdio.h>
#else
#include <OS.h>
#endif

#include "parport.h"

void PPA_TRACE(char* s);


/*----------------------------------------------------------*/
static void udelay(bigtime_t delay)
	{
	snooze(delay);
	}

/*----------------------------------------------------------*/
void* allocMem(const char* name,unsigned long size)
	{
	void* addr = NULL;
	create_area(name,&addr,B_ANY_KERNEL_ADDRESS,B_PAGE_SIZE,B_CONTIGUOUS,B_READ_AREA|B_WRITE_AREA);
	return(addr);	
	}


/*----------------------------------------------------------*/
void parport_pc_write_data(struct parport* pb,unsigned char d)
	{
	write_io_8(pb->base+DATA,d);
	}

/*----------------------------------------------------------*/
unsigned char parport_pc_read_data(struct parport* pb)
	{
	return(read_io_8(pb->base+DATA));
	}

/*----------------------------------------------------------*/
void parport_pc_write_control(struct parport* pb,unsigned char d)
	{
	write_io_8(pb->base+CONTROL,d);
	}

/*----------------------------------------------------------*/
unsigned char parport_pc_read_control(struct parport* pb)
	{
	return(read_io_8(pb->base+CONTROL));
	}

/*----------------------------------------------------------*/
void parport_pc_write_status(struct parport* pb,unsigned char d)
	{
	write_io_8(pb->base+STATUS,d);
	}

/*----------------------------------------------------------*/
unsigned char parport_pc_read_status(struct parport* pb)
	{
	return(read_io_8(pb->base+STATUS));
	}

/*----------------------------------------------------------*/
void parport_pc_write_econtrol(struct parport* pb,unsigned char d)
	{
	write_io_8(pb->base+ECONTROL,d);
	}

/*----------------------------------------------------------*/
unsigned char parport_pc_read_econtrol(struct parport* pb)
	{
	return(read_io_8(pb->base+ECONTROL));
	}

/*----------------------------------------------------------*/
void parport_pc_write_fifo(struct parport* pb,unsigned char d)
	{
	write_io_8(pb->base+CONFIGA,d);
	}

/*----------------------------------------------------------*/
unsigned char parport_pc_read_fifo(struct parport* pb)
	{
	return(read_io_8(pb->base+CONFIGA));
	}

/*----------------------------------------------------------*/
void parport_pc_write_epp(struct parport* pb,unsigned char d)
	{
	write_io_8(pb->base+EPPDATA,d);
	}

/*----------------------------------------------------------*/
unsigned char parport_pc_read_epp(struct parport* pb)
	{
	return(read_io_8(pb->base+EPPDATA));
	}




/*----------------------------------------------------------*/
int parport_pc_epp_clear_timeout(struct parport* pb)
	{
	unsigned char r;
	
	if(!(parport_pc_read_status(pb) & 0x01))
		return(1);
		
	parport_pc_read_status(pb);
	r = parport_pc_read_status(pb);
	parport_pc_write_status(pb,r | 0x01);
	parport_pc_write_status(pb,r & 0xfe);
	r = parport_pc_read_status(pb);
	
	return(!(r & 0x01));	
	}

/*----------------------------------------------------------*/
static int parport_SPP_supported(struct parport* pb)
	{
	parport_pc_epp_clear_timeout(pb);
	
	parport_pc_write_control(pb,0x0c);
	parport_pc_write_data(pb,0xaa);
	if(parport_pc_read_data(pb) != 0xaa) 
		return(0);
		
	parport_pc_write_data(pb,0x55);
	if(parport_pc_read_data(pb) != 0x55)
		return(0);

	return(1);		
	}


/*----------------------------------------------------------*/
static int parport_ECR_present(struct parport* pb)
	{
	unsigned char r,octr = parport_pc_read_control(pb);
	unsigned char oecr = parport_pc_read_econtrol(pb);
	unsigned char tmp;
	
	r = parport_pc_read_control(pb);
	if((parport_pc_read_econtrol(pb) & 0x3) == (r & 0x3))
		{
		parport_pc_write_control(pb,r ^ 0x2);
		r = parport_pc_read_control(pb);
		if((parport_pc_read_econtrol(pb) & 0x2) == (r & 0x2))
			{
			parport_pc_write_control(pb,octr);
			return(0);
			}
		}
	if((parport_pc_read_econtrol(pb) & 0x3) != 0x1)
		return(0);
		
	parport_pc_write_econtrol(pb,0x34);
	tmp = parport_pc_read_econtrol(pb);
	
	parport_pc_write_econtrol(pb,oecr);
	parport_pc_write_control(pb,octr);
	
	if(tmp != 0x35)
		return(0);
		
	return(1);
	}


/*----------------------------------------------------------*/
static int parport_ECP_supported(struct parport* pb)
	{
	int i;
	unsigned char oecr;
	
	if(!parport_ECR_present(pb))	
		return(0);
		
	oecr = parport_pc_read_econtrol(pb);
	parport_pc_write_econtrol(pb,0xc0);
	for(i=0;i < 1024 && (parport_pc_read_econtrol(pb) & 0x01);i++)
		parport_pc_write_fifo(pb,0xaa);
		
	parport_pc_write_econtrol(pb,oecr);
	return(i == 1024) ? 0 : 1;
	}

/*----------------------------------------------------------*/
static int parport_EPP_supported(struct parport* pb)
	{
	if(!parport_pc_epp_clear_timeout(pb))
		return(0);
		
	parport_pc_write_control(pb,parport_pc_read_control(pb) | 0x20);
	parport_pc_write_control(pb,parport_pc_read_control(pb) | 0x10);
	parport_pc_epp_clear_timeout(pb);
	
	parport_pc_read_epp(pb);
	udelay(30);
	
	if(parport_pc_read_status(pb) & 0x01)
		{
		parport_pc_epp_clear_timeout(pb);
		return(1);
		}
	
	return(0);
	}

/*----------------------------------------------------------*/
static int parport_ECPEPP_supported(struct parport* pb)
	{
	int mode;
	unsigned char oecr;
	
	if(!parport_ECR_present(pb))
		return(0);
	
	oecr = parport_pc_read_econtrol(pb);
	parport_pc_write_econtrol(pb,0x80);
	
	mode = parport_EPP_supported(pb);
	parport_pc_write_econtrol(pb,oecr);
	
	return(mode ? 1 : 0);
	}

/*----------------------------------------------------------*/
static int parport_PS2_supported(struct parport* pb)
	{
	int ok = 0;
	unsigned char octr = parport_pc_read_control(pb);
	
	parport_pc_epp_clear_timeout(pb);
	parport_pc_write_control(pb,octr | 0x20);
	
	parport_pc_write_data(pb,0x55);
	if(parport_pc_read_data(pb) != 0x55)
		ok++;
		
	parport_pc_write_data(pb,0xaa);
	if(parport_pc_read_data(pb) != 0xaa)
		ok++;
		
	parport_pc_write_control(pb,octr);
	return(ok ? 1 : 0);
	}

/*----------------------------------------------------------*/
static int parport_ECPPS2_supported(struct parport* pb)
	{
	int mode;
	unsigned char oecr;
	
	if(!parport_ECR_present(pb))
		return(0);
		
	oecr = parport_pc_read_econtrol(pb);
	parport_pc_write_econtrol(pb,0x20);
	
	mode = parport_PS2_supported(pb);
	
	parport_pc_write_econtrol(pb,oecr);
	return(mode ? 1 : 0);
	}


/*----------------------------------------------------------*/
static int parport_probe_port(struct parport* pb)
	{
	int ret = 0;
	
	if(pb->base)
		{
		pb->modes |= (parport_SPP_supported(pb)    ? PARPORT_MODE_PCSPP : 0);
		pb->modes |= (parport_ECR_present(pb)      ? PARPORT_MODE_PCECR : 0);
		pb->modes |= (parport_ECP_supported(pb)    ? PARPORT_MODE_PCECP : 0);
		pb->modes |= (parport_EPP_supported(pb)    ? PARPORT_MODE_PCEPP : 0);
		pb->modes |= (parport_ECPEPP_supported(pb) ? PARPORT_MODE_PCECPEPP : 0);
		pb->modes |= (parport_PS2_supported(pb)    ? PARPORT_MODE_PCPS2 : 0);
		pb->modes |= (parport_ECPPS2_supported(pb) ? PARPORT_MODE_PCECPPS2 : 0);
		
		ret = 1;
		}
	
	return(ret);
	}


/*----------------------------------------------------------*/
struct parport* parport_getarray(void)
	{
	area_id area;
	area_info info;
	struct parport* port = NULL;
	
	if((area = find_area("zip100")) == B_NAME_NOT_FOUND)
		port = allocMem("zip100",sizeof(struct parport));
	else
		{
		get_area_info(area,&info);
		port = info.address;
		}		
		
	return(port);
	}


/*----------------------------------------------------------*/
int parport_init(void)
	{
	int ret = 0;
	struct parport* port = parport_getarray();

	if(port)
		{
		memset(port,0,sizeof(struct parport));

		port->base = DEFAULT_IOPORT;
		port->name = "zip100";

		if((port->lock_sem = create_sem(1,"zip100")) >= B_NO_ERROR)
			if(set_sem_owner(port->lock_sem,B_SYSTEM_TEAM) == B_NO_ERROR)
				{
				parport_claimport(port);
				ret = parport_probe_port(port);
				parport_releaseport(port);
				}
		}
	return(ret);
	}
	
	
/*----------------------------------------------------------*/
struct parport* parport_getport(int index)
	{
	return(parport_getarray());
	}	
	
	
/*----------------------------------------------------------*/
void parport_claimport(struct parport* port)
	{
	if(port)
		acquire_sem(port->lock_sem);
	}
	
	
/*----------------------------------------------------------*/
void parport_releaseport(struct parport* port)
	{
	if(port)
		release_sem(port->lock_sem);
	}
	
	