/*
	Copyright (c) 2002, Thomas Kurschel
	based on Screen Preferences Panel written for OpenBeOS by Rafael Romo


	Part of Radeon Screen Preferences
		
	Main window, handling mode select and set
*/

#include <InterfaceDefs.h>
#include <Application.h>
#include <MenuItem.h>
#include <Window.h>
#include <Button.h>
#include <String.h>
#include <Screen.h>
//#include <Messenger.h>
#include <Menu.h>

//#include <cstring>
//#include <cstdlib>

#include <Alert.h>

#include "ScreenWindow.h"
#include "ScreenDrawView.h"
#include "AlertWindow.h"
#include "Constants.h"

#include <stdlib.h>
#include <math.h>
#include <stdio.h>

#include "multimon.h"

// list of officially supported colour spaces
struct {
	uint32 space;
	int bpp;
} bpp_list[] = {
	{ B_CMAP8, 8 },
	{ B_RGB15, 15 },
	{ B_RGB16, 16 },
	{ B_RGB32, 32 }
};

// list of standard refresh rates
struct {
	int refresh;
} refresh_list[] = {
	{56}, {60}, {70}, {72}, {75}
};

// list of combine modes
struct {
	CombineMode mode;
	const char *name;
} combine_list[] = {
	{ cb_disable, "disable" },
	{ cb_horizontally, "horizontally" },
	{ cb_vertically, "vertically" }
};

// define to use fixed standard refresh rates
// undefine to get standard refresh rates from driver
#define USE_FIXED_REFRESH

static CombineMode getCombineMode( display_mode *mode )
{
	if( mode->flags & B_SCROLL == 0 )
		return cb_disable;
		
	if( mode->virtual_width == mode->timing.h_display * 2 )
		return cb_horizontally;

	if( mode->virtual_height == mode->timing.v_display * 2 )
		return cb_vertically;
		
	return cb_disable;
}

// helper to sort modes by resolution
static int compare_mode( const void *op1, const void *op2 )
{
	display_mode
		*mode1 = (display_mode *)op1,
		*mode2 = (display_mode *)op2;
	CombineMode combine1, combine2;
	uint16 width1, width2, height1, height2;
	
	combine1 = getCombineMode( mode1 );
	combine2 = getCombineMode( mode2 );
	
	width1 = mode1->virtual_width;
	height1 = mode1->virtual_height;
	width2 = mode2->virtual_width;
	height2 = mode2->virtual_height;
	
	if( combine1 == cb_horizontally )
		width1 /= 2;
	if( combine1 == cb_vertically )
		height1 /= 2;
	if( combine2 == cb_horizontally )
		width2 /= 2;
	if( combine2 == cb_vertically )
		height2 /= 2;

	if( width1 != width2 )
		return width1 - width2;

	if( height1 != height2 )
		return height1 - height2;
		
	return ScreenWindow::getModeRefresh10( mode1 ) - 
		ScreenWindow::getModeRefresh10( mode2 );
}

BString TVStandard2Name(
	uint32 mode )
{
	switch( mode ) {
	case 0:	return "disabled";
	case 1:	return "NTSC";
	case 2:	return "NTSC Japan";
	case 3:	return "PAL BDGHI";
	case 4:	return "PAL M";
	case 5:	return "PAL N";
	case 6:	return "SECAM";
	case 101:	return "NTSC 443";
	case 102:	return "PAL 60";
	case 103:	return "PAL NC";
	default:	{
		BString res;
		
		res << "??? (" << mode << ")";
		
		return res; }
	}
}

// create all elements of main window
ScreenWindow::ScreenWindow(ScreenSettings *Settings)
	: BWindow(Settings->WindowFrame(), "Radeon Screen", B_TITLED_WINDOW, B_NOT_RESIZABLE | B_NOT_ZOOMABLE, B_ALL_WORKSPACES)
{
	BScreen screen( this );
	BRect frame = Bounds();
	
	BView *view;
	
	view = new BView( frame, "ScreenView", B_FOLLOW_ALL, B_WILL_DRAW );
	view->SetViewColor( ui_color( B_PANEL_BACKGROUND_COLOR ));
	AddChild( view );
	
	fSettings = Settings;
	fChangingAllWorkspaces = false;
	
	BRect ButtonRect;


	// area with screen preview and workspace count	
	{
		BRect ScreenBoxRect;
		BRect ScreenDrawViewRect;
		BRect WorkspaceCountMenuRect;
		
		ScreenBoxRect.Set(11.0, 18.0, 153.0, 155.0);
		
		fScreenBox = new BBox(ScreenBoxRect);
		fScreenBox->SetBorder(B_FANCY_BORDER);
		
		ScreenDrawViewRect.Set(20.0, 16.0, 122.0, 93.0);
		
		fScreenDrawView = new ScreenDrawView(ScreenDrawViewRect, "ScreenDrawView");
		
		int32 Count;
		
		BString String;
		
		String << count_workspaces();
		
		fWorkspaceCountMenu = new BPopUpMenu(String.String(), true, true);
		for (Count = 1; Count <= 32; Count++)
		{
			String.Truncate(0);
			String << Count;
			fWorkspaceCountMenu->AddItem(new BMenuItem(String.String(), new BMessage(POP_WORKSPACE_CHANGED_MSG)));
		}
		
		String.Truncate(0);
		String << count_workspaces();
		
		BMenuItem *Marked = fWorkspaceCountMenu->FindItem(String.String());
		Marked->SetMarked(true);
		
		WorkspaceCountMenuRect.Set(7.0, 107.0, 135.0, 127.0);
		
		fWorkspaceCountField = new BMenuField(WorkspaceCountMenuRect, "WorkspaceCountMenu", "Workspace count:", fWorkspaceCountMenu, true);
		
		fWorkspaceCountField->SetDivider(91.0);
		
		fScreenBox->AddChild(fScreenDrawView);
		
		fScreenBox->AddChild(fWorkspaceCountField);
		
		view->AddChild(fScreenBox);
	}
	
	
	// area with screen resolution etc.
	{
		BRect ControlsBoxRect;
		BRect WorkspaceMenuRect;
		BRect ControlMenuRect;

		fWorkspaceMenu = new BPopUpMenu("Current Workspace", true, true);
		fAllWorkspacesItem = new BMenuItem("All Workspaces", new BMessage(WORKSPACE_CHECK_MSG));
		fWorkspaceMenu->AddItem(fAllWorkspacesItem);
		fCurrentWorkspaceItem = new BMenuItem("Current Workspace", new BMessage(WORKSPACE_CHECK_MSG));
		fCurrentWorkspaceItem->SetMarked(true);
		fWorkspaceMenu->AddItem(fCurrentWorkspaceItem);
		
		WorkspaceMenuRect.Set(0.0, 0.0, 132.0, 18.0);
		
		fWorkspaceField = new BMenuField(WorkspaceMenuRect, "WorkspaceMenu", NULL, fWorkspaceMenu, true);
		
		ControlsBoxRect.Set(164.0, 7.0, 345.0, 155.0);
		
		fControlsBox = new BBox(ControlsBoxRect);
		fControlsBox->SetBorder(B_FANCY_BORDER);
		fControlsBox->SetLabel(fWorkspaceField);
		
		ButtonRect.Set(88.0, 114.0, 200.0, 150.0);
		
		fApplyButton = new BButton(ButtonRect, "ApplyButton", "Apply", 
		new BMessage(BUTTON_APPLY_MSG));
		
		fApplyButton->AttachedToWindow();
		fApplyButton->ResizeToPreferred();
		fApplyButton->SetEnabled(false);
		
		fControlsBox->AddChild(fApplyButton);
		
		screen.GetModeList( &fModeList, &fModeListCount );
	
		// sort modes by resolution and refresh to make 
		// the resolution and refresh menu look nicer
		qsort( fModeList, fModeListCount, sizeof( fModeList[0] ), compare_mode );
		
		fResolutionMenu = new BPopUpMenu("", true, true);
		
		uint16 prev_width, prev_height;
		uint i;
			
		prev_width = 0;
		prev_height = 0;
		
		for( i = 0; i < fModeListCount; ++i ) {
			uint16 width, height;
			CombineMode combine;
			
			width = fModeList[i].virtual_width;
			height = fModeList[i].virtual_height;
			
			combine = getCombineMode( &fModeList[i] );
			
			if( combine == cb_horizontally )
				width /= 2;
			if( combine == cb_vertically )
				height /= 2;
			
			if( width != prev_width || height != prev_height )
			{
				BString name;
				BMessage *msg;
				
				prev_width = width;
				prev_height = height;
				
				name << (uint)width << " x " << (uint)height;
				
				msg = new BMessage( POP_RESOLUTION_MSG );
				msg->AddInt32( "width", width );
				msg->AddInt32( "height", height );
				
				fResolutionMenu->AddItem( new BMenuItem( name.String(), msg ));
			}
		}
	
		ControlMenuRect.Set(33.0, 30.0, 171.0, 48.0);
		
		fResolutionField = new BMenuField(ControlMenuRect, "ResolutionMenu", "Resolution:", fResolutionMenu, true);
		
		fResolutionField->SetDivider(55.0);
		
		fControlsBox->AddChild(fResolutionField);
		
		fColorsMenu = new BPopUpMenu("", true, true);
			
		for( i = 0; i < sizeof( bpp_list ) / sizeof( bpp_list[0] ); ++i ) {
			BString name;
			BMessage *msg;
			
			name << bpp_list[i].bpp << " Bits/Pixel";
			
			msg = new BMessage( POP_COLORS_MSG );
			msg->AddInt32( "bpp", bpp_list[i].bpp );
			msg->AddInt32( "space", bpp_list[i].space );
			
			fColorsMenu->AddItem( new BMenuItem( name.String(), msg ));
		}
			
		ControlMenuRect.Set(50.0, 58.0, 171.0, 76.0);
		
		fColorsField = new BMenuField(ControlMenuRect, "ColorsMenu", "Colors:", fColorsMenu, true);
		
		fColorsField->SetDivider(38.0);
		
		fControlsBox->AddChild(fColorsField);
		
		
		fRefreshMenu = new BPopUpMenu("", true, true);
		
	#ifdef USE_FIXED_REFRESH
		for( i = 0; i < sizeof( refresh_list ) / sizeof( refresh_list[0] ); ++i ) {
			BString name;
			BMessage *msg;
			
			name << refresh_list[i].refresh << " Hz";
			
			msg = new BMessage( POP_REFRESH_MSG );
			msg->AddInt32( "refresh10", refresh_list[i].refresh * 10 );
			
			fRefreshMenu->AddItem( new BMenuItem( name.String(), msg ));
		}
	#endif
	
		BMessage *msg;
		
		msg = new BMessage( POP_OTHER_REFRESH_MSG );
		msg->AddInt32( "refresh10", 0 );
		
		fOtherRefresh = new BMenuItem( "Other...", msg );
		fRefreshMenu->AddItem( fOtherRefresh );
	
		ControlMenuRect.Set(19.0, 86.0, 171.0, 104.0);
		
		fRefreshField = new BMenuField(ControlMenuRect, "RefreshMenu", "Refresh Rate:", fRefreshMenu, true);
		
		fRefreshField->SetDivider(69.0);
		
		fControlsBox->AddChild(fRefreshField);
		
		view->AddChild(fControlsBox);
	}
	
	
	// enlarged area for multi-monitor settings
	{
		BRect rect;
		BMenuField *menuField;
		BMenuItem *item;
		BMessage *msg;
		uint i;
		bool dummy;
		uint32 dummy32;
		bool multiMonSupport;
		bool useLaptopPanelSupport;
		bool tvStandardSupport;

		multiMonSupport = TestMultiMonSupport( &screen ) == B_OK;
		useLaptopPanelSupport = GetUseLaptopPanel( &screen, &dummy ) == B_OK;
		tvStandardSupport = GetTVStandard( &screen, &dummy32 ) == B_OK;
		
		// even if there is no support, we still create all controls
		// to make sure we don't access NULL pointers later on
		if( multiMonSupport ) {
			fApplyButton->MoveTo( 275, 114 );
			fControlsBox->ResizeTo( 366, 148 );		
			ResizeTo( 556, 202 );
		}
		
		fCombineMenu = new BPopUpMenu( "CombineDisplays", true, true );

		for( i = 0; i < sizeof( combine_list ) / sizeof( combine_list[0] ); ++i ) {
			BMessage *msg;
			BMenuItem *item;
			
			msg = new BMessage( POP_COMBINE_DISPLAYS_MSG );
			msg->AddInt32( "mode", combine_list[i].mode );
			item = new BMenuItem( combine_list[i].name, msg );
			fCombineMenu->AddItem( item );
		}
		
		rect.Set( 185, 30, 356, 48 );
		menuField = new BMenuField( rect, "CombineMenu", "Combine Displays:", 
			fCombineMenu, true );
		menuField->SetDivider( 90 );
		
		fControlsBox->AddChild( menuField );
		if( !multiMonSupport )
			menuField->Hide();

		fSwapDisplaysMenu = new BPopUpMenu( "SwapDisplays", true, true );

		// !order is important - we rely that boolean value == idx
		msg = new BMessage( POP_SWAP_DISPLAYS_MSG );
		msg->AddBool( "swap", false );
		item = new BMenuItem( "no", msg );
		fSwapDisplaysMenu->AddItem( item );
			
		msg = new BMessage( POP_SWAP_DISPLAYS_MSG );
		msg->AddBool( "swap", true );
		item = new BMenuItem( "yes", msg );
		fSwapDisplaysMenu->AddItem( item );
		
		rect.Set( 199, 58, 356, 76 );
		menuField = new BMenuField( rect, "SwapMenu", "Swap Displays:", 
			fSwapDisplaysMenu, true );
		menuField->SetDivider( 76 );

		fControlsBox->AddChild( menuField );
		if( !multiMonSupport )
			menuField->Hide();

		fUseLaptopPanelMenu = new BPopUpMenu( "UseLaptopPanel", true, true );

		// !order is important - we rely that boolean value == idx			
		msg = new BMessage( POP_USE_LAPTOP_PANEL_MSG );
		msg->AddBool( "use", false );
		item = new BMenuItem( "if needed", msg );
		fUseLaptopPanelMenu->AddItem( item );
		
		msg = new BMessage( POP_USE_LAPTOP_PANEL_MSG );
		msg->AddBool( "use", true );
		item = new BMenuItem( "always", msg );
		fUseLaptopPanelMenu->AddItem( item );

		rect.Set( 184, 86, 356, 104 );

		menuField = new BMenuField( rect, "UseLaptopPanel", "Use Laptop Panel:", 
			fUseLaptopPanelMenu, true );
		menuField->SetDivider( 91 );

		fControlsBox->AddChild( menuField );
		if( !useLaptopPanelSupport )
			menuField->Hide();
		
			
		fTVStandardMenu = new BPopUpMenu( "TVStandard", true, true );
		
		// arbitrary limit
		for( i = 0; i < 100; ++i ) {
			uint32 mode;
			
			if( GetNthSupportedTVStandard( &screen, i, &mode ) != B_OK )
				break;
			
			BString name = TVStandard2Name( mode );
			
			BMessage *msg;
			
			msg = new BMessage( POP_TV_STANDARD_MSG );
			msg->AddInt32( "tv_standard", mode );
			
			fTVStandardMenu->AddItem( new BMenuItem( name.String(), msg ));
		}
			
		rect.Set( 15, 114, 171, 132 );
		
		menuField = new BMenuField( rect, "TVStandard", "Video Format:", 
			fTVStandardMenu, true );
		
		menuField->SetDivider( 73 );

		fControlsBox->AddChild( menuField );
		if( !tvStandardSupport || i == 0 )
			menuField->Hide();
	}
		
	ButtonRect.Set(10.0, 167, 100.0, 200.0);
	
	fDefaultsButton = new BButton(ButtonRect, "DefaultsButton", "Defaults", 
	new BMessage(BUTTON_DEFAULTS_MSG));
	
	fDefaultsButton->AttachedToWindow();
	fDefaultsButton->ResizeToPreferred();
	
	view->AddChild(fDefaultsButton);
	
	ButtonRect.Set(95.0, 167, 160.0, 200.0);
	
	fRevertButton = new BButton(ButtonRect, "RevertButton", "Revert", 
	new BMessage(BUTTON_REVERT_MSG));
	
	fRevertButton->AttachedToWindow();
	fRevertButton->ResizeToPreferred();
	fRevertButton->SetEnabled(false);
	
	view->AddChild(fRevertButton);

	fOrigSwapDisplays = false;
	GetSwapDisplays( &screen, &fOrigSwapDisplays );
	fOrigUseLaptopPanel = false;
	GetUseLaptopPanel( &screen, &fOrigUseLaptopPanel );
	fOrigTVStandard = 0;
	GetTVStandard( &screen, &fOrigTVStandard );
	
	screen.GetMode( &fOrigMode );
	ReflectActiveMode();
	
	Show();
}

ScreenWindow::~ScreenWindow()
{
	delete fSettings;
}

// get refresh rate of mode in 1/10 Hz (this accuracy is used everywhere)
int ScreenWindow::getModeRefresh10( display_mode *mode )
{
	float fRefresh;
	int refresh10;
	
	// we have to be catious as refresh rate cannot be controlled directly,
	// so it suffers under rounding errors and hardware restrictions
	fRefresh = float(mode->timing.pixel_clock * 1000) / 
		float(mode->timing.h_total * mode->timing.v_total);
	
	refresh10 = (int)rint( fRefresh * 10 );

	return refresh10;
}

// update resolution list according to combine mode
// (some resolution may not be combinable due to memory restrictions)
void ScreenWindow::UpdateResolutions( CombineMode combine )
{		
	for( int i = 0; i < fResolutionMenu->CountItems(); ++i ) 
		fResolutionMenu->ItemAt( i )->SetEnabled( false );
	
	for( uint j = 0; j < fModeListCount; ++j ) {
		uint16 width, height;
		BMenuItem *item;
		
		if( getCombineMode( &fModeList[j] ) != combine )
			continue;
		
		width = fModeList[j].virtual_width;
		height = fModeList[j].virtual_height;
		
		BString name;
			
		name 
			<< (uint)(combine == cb_horizontally ? width / 2 : width)
			<< " x " 
			<< (uint)(combine == cb_vertically ? height / 2 : height);
			
		item = fResolutionMenu->FindItem( name.String() );
		if( item != NULL )
			item->SetEnabled( true );
	}
}


// update bpp and refresh options according to current mode
// (a colour space is made active if there is any mode with 
//  given resolution and this colour space; same applies for 
//  refresh rate, though "other..." is always possible)
void ScreenWindow::UpdateModeOptions( int width, int height, CombineMode combine )
{	
	uint i, j;
	uint16 virtual_width, virtual_height;
	
	virtual_width = combine == cb_horizontally ? width * 2 : width;
	virtual_height = combine == cb_vertically ? height * 2 : height;

	for( i = 0; i < sizeof( bpp_list ) / sizeof( bpp_list[0] ); ++i ) {
		BMenuItem *item;
		bool supported = false;
		
		for( j = 0; j < fModeListCount; ++j ) {
			if( virtual_width == fModeList[j].virtual_width &&
				virtual_height == fModeList[j].virtual_height &&
				bpp_list[i].space == fModeList[j].space &&
				combine == getCombineMode( &fModeList[j] ))
			{
				supported = true;
				break;
			}
		}

		item = fColorsMenu->ItemAt( i );		
		if( item )
			item->SetEnabled( supported );
	}
	
#ifndef USE_FIXED_REFRESH	
	for( i = fRefreshMenu->CountItems() - 2; i >= 0; --i ) {
		BMenuItem *item;
		
		item = fRefreshMenu->RemoveItem( i );
		if( item )
			delete item;
	}

	for( i = 0; i < fModeListCount; ++i ) {
		if( virtual_width == fModeList[i].virtual_width &&
			virtual_width == fModeList[i].virtual_height &&
			combine == getCombineMode( &fModeList[i] ))
		{
			BString name;
			BMenuItem *item;
			int refresh10;
			
			refresh10 = getModeRefresh10( &fModeList[i] );
			name << Refresh2Text( refresh10, false ) << " Hz";
			
			item = fRefreshMenu->FindItem( name.String() );
			if( item == NULL ) {
				BMessage *msg;
				
				msg = new BMessage( POP_REFRESH_MSG );
				msg->AddInt32( "refresh10", refresh10 );
				
				fRefreshMenu->AddItem( 
					new BMenuItem( name.String(), msg ), 
					fRefreshMenu->CountItems() - 1 );
			}
		}
	}
#endif			

	// TBD: some drivers lack many refresh rates; still, they
	// can be used by generating the mode manually
/*
	for( i = 0; i < sizeof( refresh_list ) / sizeof( refresh_list[0] ); ++i ) {
		BMenuItem *item;
		bool supported = false;
		
		for( j = 0; j < fModeListCount; ++j ) {
			if( width == fModeList[j].virtual_width &&
				height == fModeList[j].virtual_height &&
				refresh_list[i].refresh * 10 == getModeRefresh10( &fModeList[j] ))
			{
				supported = true;
				break;
			}
		}

		item = fRefreshMenu->ItemAt( i );
		if( item ) 
			item->SetEnabled( supported );
	}
*/
}

BString ScreenWindow::Refresh2Text( int refresh10, bool always_with_fraction )
{
	if( always_with_fraction || refresh10 % 10 != 0 ) {
		char buffer[100];

		// I always distrust sprintf, especially with 
		// floating point (if refresh10 is something like 1e4000,
		// we get a *very* long output; as refresh10 is int, i.e.
		// in [-2e9,2e9], the result should consist of up to
		// 8 significant digits and 1 fractional digit, altogether
		// 12 character; anyway - I don't trust these estimations,
		// something link snprintf made me sleep more easily)
		sprintf( buffer, "%3.1f", refresh10 / 10.0 );
		return buffer;
	} else {
		BString str;
		
		str << refresh10 / 10;
		return str;
	}
}

// activate appropriate menu item according to selected refresh rate
void ScreenWindow::ReflectSelectedRefresh()
{
	BString str, str2;
	BMenuItem *item;
	
	item = NULL;

	// try to find refresh rate in list of standard refreshes
/*	// (don't do that if refresh has fractional part as this
	//  gets discarded before searching)
	if( fSelectedRefresh10 % 10 == 0 ) {
		// we are lazy and ask refresh menu directly instead of
		// going through refresh_list
		str.Truncate( 0 );
		str << fSelectedRefresh10 / 10 << " Hz";
		
		item = fRefreshMenu->FindItem( str.String() );
	}*/
	
	str2 = Refresh2Text( fSelectedRefresh10, false );
	str.Truncate( 0 );
	str << str2 << " Hz";
	
	item = fRefreshMenu->FindItem( str.String() );

	if( item ) {
		item->SetMarked( true );
		// "Other..." items only contains a refresh rate when active
		fOtherRefresh->SetLabel( "Other..." );
		return;
	} 
	
	// this is a non-standard refresh rate
	fOtherRefresh->Message()->ReplaceInt32( "refresh10", (int)fSelectedRefresh10 );

	str2 = Refresh2Text( fSelectedRefresh10, true );	
	str.Truncate( 0 );
	str << str2 << " Hz/Other...";
	
	fOtherRefresh->SetLabel( str.String() );
	fOtherRefresh->SetMarked( true );

	// we don't want this "Other..." appear in menu label
	str.Truncate( 0 );
	str << str2 << " Hz";
	
	fRefreshMenu->Superitem()->SetLabel( str.String() );
}

void ScreenWindow::ReflectSelectedResolution()
{
	BMessage *Message = new BMessage(UPDATE_DESKTOP_MSG);
	
	Message->AddInt32( "width", fSelectedWidth );
	Message->AddInt32( "height", fSelectedHeight );

	PostMessage( Message, fScreenDrawView );
}

// reflect active mode in chosen settings
void ScreenWindow::ReflectActiveMode()
{
	BScreen screen( this );
	display_mode mode;
	BString str;
	BMenuItem *item;
	int i;

	// usually, this function gets called after a mode
	// has been set manually; still, as the graphics driver
	// is free to fiddle with mode passed, we better ask 
	// what kind of mode we actually got
	screen.GetMode( &mode );
	
	fActiveMode = mode;
	
	fActiveWidth = mode.virtual_width;
	fActiveHeight = mode.virtual_height;
	fSelectedSpace = fActiveSpace = mode.space;
	fSelectedRefresh10 = fActiveRefresh10 = getModeRefresh10( &mode );
	fSelectedCombineMode = fActiveCombineMode = getCombineMode( &mode );
	
	if( fActiveCombineMode == cb_horizontally )
		fActiveWidth /= 2;
	if( fActiveCombineMode == cb_vertically )
		fActiveHeight /= 2;
		
	fSelectedWidth = fActiveWidth;
	fSelectedHeight = fActiveHeight;
	
	// assign some value so we GetSwapDisplays,... can fail silently
	// (else, "Apply" and "Revert" is always enabled if SwapDisplay-variables differ)
	fActiveSwapDisplays = fOrigSwapDisplays;
	GetSwapDisplays( &screen, &fActiveSwapDisplays );
	fSelectedSwapDisplays = fActiveSwapDisplays;
	
	fActiveUseLaptopPanel = fOrigUseLaptopPanel;
	GetUseLaptopPanel( &screen, &fActiveUseLaptopPanel );
	fSelectedUseLaptopPanel = fActiveUseLaptopPanel;
	
	fActiveTVStandard = fOrigTVStandard;
	GetTVStandard( &screen, &fActiveTVStandard );
	fSelectedTVStandard = fActiveTVStandard;
	
	item = fSwapDisplaysMenu->ItemAt( fSelectedSwapDisplays );
	if( item )
		item->SetMarked( true );
	
	item = fUseLaptopPanelMenu->ItemAt( fSelectedUseLaptopPanel );
	if( item )
		item->SetMarked( true );
	
	for( i = 0; i < fTVStandardMenu->CountItems(); ++i ) {
		uint32 cur_tv_standard;
		
		item = fTVStandardMenu->ItemAt( i );
		item->Message()->FindInt32( "tv_standard", (int32 *)&cur_tv_standard );
		if( cur_tv_standard == fSelectedTVStandard ) {
			item->SetMarked( true );
			break;
		}
	}

	UpdateResolutions( fActiveCombineMode );
	UpdateModeOptions( fActiveWidth, fActiveHeight, fActiveCombineMode );
	
	str.Truncate( 0 );
	str << (uint)fActiveWidth << " x " << (uint)fActiveHeight;
	item = fResolutionMenu->FindItem( str.String() );
	
	/*item = NULL;

	for( i = 0; i < fResolutionMenu->CountItems(); ++i ) {
		BMenuItem *cur_item;
		BMessage *msg;
		int32 tmp_width, tmp_height;
		
		cur_item = fResolutionMenu->ItemAt( i );
		msg = cur_item->Message();
		if( msg->FindInt32( "width", &tmp_width ) != B_OK ||
			msg->FindInt32( "height", &tmp_height ) != B_OK )
			continue;
			
		if( fActiveWidth == mode.virtual_width && fActiveHeight == mode.virtual_height ) {
			item = cur_item;
			break;
		}
	}*/
	
	if( item != NULL ) {
		item->SetMarked( true );
	} else {
		// this is bad luck - if mode has been set via screen references, 
		// this case cannot occur; there are three possible solutions:
		// 1. add a new resolution to list
		//    - we had to remove it as soon as a "valid" one is selected
		//    - we don't know which frequencies/bit depths are supported
		//    - as long as we haven't the GMT formula to create 
		//      parameters for any resolution given, we cannot
		//      really set current mode - it's just not in the list
		// 2. choose nearest resolution
		//    - probably a good idea, but implies coding and testing
		// 3. choose lowest resolution
		//    - do you really think we are so lazy? yes, we are
		item = fResolutionMenu->ItemAt( 0 );
		if( item )
			item->SetMarked( true );

		// okay - at least we set menu label to active resolution
		fResolutionMenu->Superitem()->SetLabel( str.String() );
	}

	// mark active combine mode
	for( i = 0; (uint)i < sizeof( combine_list ) / sizeof( combine_list[0] ); ++i )
		if( combine_list[i].mode == fActiveCombineMode )
			break;
			
	item = fCombineMenu->ItemAt( i );
	// paranoia
	if( item )
		item->SetMarked( true );

	for( i = sizeof( bpp_list ) / sizeof( bpp_list[0] ) - 1; i >= 0; --i ) {
		if( bpp_list[i].space == mode.space )
			break;
	}

	// fall back to 8 bits if colour space is unknown 
	// (shouldn't happen as we know all colour spaces the 
	//  application server supports)
	if( i < 0 )
		i = 0;
	
	item = fColorsMenu->ItemAt( i );
	// well - this test is paranoid, but - who knows 8|
	if( item )
		item->SetMarked( true );

	fSelectedBpp = fActiveBpp = bpp_list[i].bpp;

	ReflectSelectedResolution();
	ReflectSelectedRefresh();

	CheckApplyEnabled();
}

bool ScreenWindow::QuitRequested()
{
	be_app->PostMessage(B_QUIT_REQUESTED);
	
	return(true);
}

void ScreenWindow::FrameMoved(BPoint position)
{
	fSettings->SetWindowFrame(Frame());
}

void ScreenWindow::ScreenChanged(BRect frame, color_space mode)
{
	if (frame.right <= Frame().right
				&& frame.bottom <= Frame().bottom)
		MoveTo(((frame.right / 2) - 178), ((frame.right / 2) - 101));
}

// if another workspace gets activated, we forget everything
// and start from scratch
void ScreenWindow::WorkspaceActivated( int32 ws, bool active )
{
	if( fChangingAllWorkspaces )
		return;
		
	BScreen( this ).GetMode( &fOrigMode );
	
	ReflectActiveMode();
	
	PostMessage(new BMessage(UPDATE_DESKTOP_COLOR_MSG), fScreenDrawView);
}

// there is something to apply if active and selected mode differ
bool ScreenWindow::canApply()
{
	if( fSelectedWidth != fActiveWidth || fSelectedHeight != fActiveHeight 
		|| fSelectedSpace != fActiveSpace || fSelectedRefresh10 != fActiveRefresh10 
		|| fSelectedCombineMode != fActiveCombineMode 
		|| fSelectedSwapDisplays != fActiveSwapDisplays 
		|| fSelectedUseLaptopPanel != fActiveUseLaptopPanel 
		|| fSelectedTVStandard != fActiveTVStandard )
		return true;

	return false;
}

// as reverting means to falling-back to mode encountered
// at program start (aka workspace switch), we enable this
// button if active mode (fPrev...) and original mode differ
// or if some settings have been changed by user
bool ScreenWindow::canRevert()
{
	if( memcmp( &fActiveMode, &fOrigMode, sizeof( display_mode )) != 0 )
		return true;
		
	if( canApply() )
		return true;
		
	return false;
}

// enable/disable "apply" and "revert" buttons
void ScreenWindow::CheckApplyEnabled()
{
	fRevertButton->SetEnabled( canRevert() );
	fApplyButton->SetEnabled( canApply() );
}

// extract selected mode to "mode", return true = success
bool ScreenWindow::getSelectedMode( display_mode *mode )
{
	uint i;
	uint16 virtual_width, virtual_height;
	
	virtual_width = fSelectedCombineMode == cb_horizontally ? 
		fSelectedWidth * 2 : fSelectedWidth;
	virtual_height = fSelectedCombineMode == cb_vertically ? 
		fSelectedHeight * 2 : fSelectedHeight;
	
	// try to find mode in list provided by driver
	for( i = 0; i < fModeListCount; ++i ) {
		if( fModeList[i].virtual_width == virtual_width &&
			fModeList[i].virtual_height == virtual_height &&
			fModeList[i].space == fSelectedSpace &&
			getModeRefresh10( &fModeList[i] ) == fSelectedRefresh10 )
		{
			break;
		}
	}
	
	if( i < fModeListCount ) {
		// we have luck - use this mode directly
		*mode = fModeList[i];
		mode->h_display_start = 0;
		mode->v_display_start = 0;
		return true;
	}

	// now, we are better of using GMT formula, but
	// as we don't have it, we look for a mode with
	// same resultion and colour space and a similar
	// refresh rate and tweak its pixel clock
	int best_idx = -1;
	int best_diff = INT_MAX;
	
	for( i = 0; i < fModeListCount; ++i ) {
		if( fModeList[i].virtual_width == virtual_width &&
			fModeList[i].virtual_height == virtual_height &&
			fModeList[i].space == fSelectedSpace ) 
		{
			int diff;
			
			diff = abs( getModeRefresh10( &fModeList[i] ) - fSelectedRefresh10 );
			if( diff < best_diff ) {
				best_diff = diff;
				best_idx = i;
			}
		}
	}
	
	if( best_idx < 0 )
		return false;
		
	*mode = fModeList[best_idx];

	// after some fiddling, it looks like this is the formula
	// used by the original panel (notice that / 10 happens before
	// multiplying with refresh rate - this leads to different
	// rounding)
	mode->timing.pixel_clock = 
		((uint32)mode->timing.h_total * mode->timing.v_total / 10 
		* fSelectedRefresh10) / 1000;
		
	mode->h_display_start = 0;
	mode->v_display_start = 0;
	
	return true;
}

// apply selected mode
void ScreenWindow::Apply()
{
	BScreen screen( B_MAIN_SCREEN_ID );
	display_mode mode;
	
	if( !getSelectedMode( &mode ))
		return;
	
	if( fWorkspaceMenu->FindMarked() == fAllWorkspacesItem )	{
		BAlert *WorkspacesAlert = new BAlert("WorkspacesAlert", "Change all workspaces? 
This action cannot be reverted", "Okay", "Cancel", 
			NULL, B_WIDTH_AS_USUAL, B_WARNING_ALERT);
	
		if( WorkspacesAlert->Go() == 1 )
			return;
	}
	
	SetSwapDisplays( &screen, fSelectedSwapDisplays );
	SetUseLaptopPanel( &screen, fSelectedUseLaptopPanel );
	SetTVStandard( &screen, fSelectedTVStandard );
	screen.SetMode( &mode, true );
		
	BRect Rect;

	Rect.Set(100.0, 100.0, 400.0, 193.0);
	
	Rect.left = (screen.Frame().right / 2) - 150;
	Rect.top = (screen.Frame().bottom / 2) - 42;
	Rect.right = Rect.left + 300.0;
	Rect.bottom = Rect.top + 93.0;
	
	new AlertWindow( this, Rect);
}

void ScreenWindow::MessageReceived(BMessage* message)
{
	switch(message->what) {
		case WORKSPACE_CHECK_MSG:
			fApplyButton->SetEnabled(true);
			break;
			
		case POP_COLORS_MSG: 
			// new colour has been chosen
			message->FindInt32( "bpp", &fSelectedBpp );
			message->FindInt32( "space", (int32 *)&fSelectedSpace );
			
			CheckApplyEnabled();
			break;
			
		case POP_REFRESH_MSG:
			// new (standard) refresh rate has been chosen
			message->FindInt32( "refresh10", &fSelectedRefresh10 );
			
			fOtherRefresh->SetLabel( "Other..." );
			
			CheckApplyEnabled();
			break;
	
		case POP_WORKSPACE_CHANGED_MSG: {		
			BMenuItem *Item = fWorkspaceCountMenu->FindMarked();
		
			set_workspace_count( fWorkspaceCountMenu->IndexOf(Item) + 1 );
			break;
		}
		
		case POP_RESOLUTION_MSG: {
			// new resolution has been chosen
			// we need to update bpp and refresh according to
			// card's capabilities and redraw screen icon
			message->FindInt32( "width", &fSelectedWidth );
			message->FindInt32( "height", &fSelectedHeight );

			ReflectSelectedResolution();

			UpdateModeOptions( fSelectedWidth, fSelectedHeight, fSelectedCombineMode );

			ReflectSelectedRefresh();
			
			CheckApplyEnabled();			
			break;
		}
		
		case POP_COMBINE_DISPLAYS_MSG: {
			int32 tmp;
			
			// new combine mode has bee chosen
			message->FindInt32( "mode", &tmp );
			
			fSelectedCombineMode = (CombineMode)tmp;
			
			UpdateResolutions( fSelectedCombineMode );
			
			CheckApplyEnabled();
			break;
		}
		
		case POP_SWAP_DISPLAYS_MSG: {
			message->FindBool( "swap", &fSelectedSwapDisplays );
			
			CheckApplyEnabled();
			break;
		}
		
		case POP_USE_LAPTOP_PANEL_MSG: {
			message->FindBool( "use", &fSelectedUseLaptopPanel );
			
			CheckApplyEnabled();
			break;
		}
		
		case POP_TV_STANDARD_MSG: {
			message->FindInt32( "tv_standard", (int32 *)&fSelectedTVStandard );
			
			CheckApplyEnabled();
			break;
		}
		
		case POP_OTHER_REFRESH_MSG: {
			// user pressed "other..." in refresh rate list;
			// take last refresh as a base 
			CheckApplyEnabled();
			
			// make sure menu shows something usefull
			ReflectSelectedRefresh();
		
			fRefreshWindow = new RefreshWindow(
				this,
				BRect(
					(Frame().left + 201.0), (Frame().top + 34.0), 
					(Frame().left + 509.0), (Frame().top + 169.0)), 
				fSelectedRefresh10 );
		
			break;
		}
		
		case BUTTON_DEFAULTS_MSG: {	
			// user pressed "default" button;
			// this chooses 640x480x8@60 but *doesn't* activate it
			BMenuItem *item;
			
			// well, every driver should support 640x480, 
			// but think about VESA 
			item = fResolutionMenu->FindItem("640 x 480");
			if( item ) {
				fSelectedWidth = 640;
				fSelectedHeight = 480;
				item->SetMarked(true);
			}
				
			fColorsMenu->FindItem("8 Bits/Pixel")->SetMarked(true);
			fSelectedBpp = 8;
			fSelectedSpace = B_CMAP8;
			
			fSelectedRefresh10 = 600;
			ReflectSelectedRefresh();
			
			CheckApplyEnabled();	
			break;
		}
		
		case BUTTON_REVERT_MSG:	{
			// user pressed "revert"
			// switch back to mode that was active when this
			// panel was started or the current workspace was
			// activated; don't ask anything, just do it
			BScreen screen( this );
			
			SetSwapDisplays( &screen, fOrigSwapDisplays );
			SetUseLaptopPanel( &screen, fOrigUseLaptopPanel );
			SetTVStandard( &screen, fOrigTVStandard );
			screen.SetMode( &fOrigMode, true );
			
			ReflectActiveMode();
			break;
		}
			
		case BUTTON_APPLY_MSG: {
			// user asks to apply the mode he has chosen;
			// after changing the mode (and a warning, if
			// the refresh is very high), a confirmation
			// box pops up, reverting to previous mode after
			// a timeout
			Apply();
			break;
		}
		
		case SET_INITIAL_MODE_MSG: {
			// user pressed "revert" in confirmation box;
			// revert to mode previously active
			BScreen screen( this );
			
			SetSwapDisplays( &screen, fActiveSwapDisplays );
			SetUseLaptopPanel( &screen, fActiveUseLaptopPanel );
			SetTVStandard( &screen, fActiveTVStandard );
			BScreen( this ).SetMode( &fActiveMode, true );
			
			ReflectActiveMode();
			break;
		}
		
		case SET_CUSTOM_REFRESH_MSG: {
			// user pressed "done" in "other..." refresh dialog;
			// select the refresh rate chosen
			message->FindInt32( "refresh10", &fSelectedRefresh10 );
			
			ReflectSelectedRefresh();
			CheckApplyEnabled();
			break;
		}
		
		case MAKE_INITIAL_MSG: {
			// user pressed "keep" in confirmation box;
			// select this mode in dialog and mark it as
			// previous mode; if "all workspaces" is selected, 
			// distribute mode to all workspaces 
	
			display_mode new_mode;
			BScreen screen( this );
					
			// use the mode that has eventually been set and
			// thus we know to be working; it can differ from 
			// the mode selected by user due to hardware limitation
			screen.GetMode( &new_mode );
			
			if (fWorkspaceMenu->FindMarked() == fAllWorkspacesItem )	{
				int32 orig, i;
								
				// the original panel activates each workspace in turn;
				// this is disguisting and there is a SetMode
				// variant that accepts a workspace id, so let's take
				// this one
				orig = current_workspace();

				// well, this "cannot be reverted" message is not
				// entirely true - at least, you can revert it
				// for current workspace; to not overwrite original
				// mode during workspace switch, we use this flag
				fChangingAllWorkspaces = true;

				for( i = 0; i < count_workspaces(); ++i ) {
					if( i != orig )
						screen.SetMode( i, &new_mode, true );
				}
			
				fChangingAllWorkspaces = false;
			} 
						
			ReflectActiveMode();
			break;
		}
		
		default:
			BWindow::MessageReceived(message);
		
			break;
	}
}
