/*
Open Tracker License

Terms and Conditions

Copyright (c) 1991-2000, Be Incorporated. All rights reserved.

Permission is hereby granted, free of charge, to any person obtaining a copy of
this software and associated documentation files (the "Software"), to deal in
the Software without restriction, including without limitation the rights to
use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies
of the Software, and to permit persons to whom the Software is furnished to do
so, subject to the following conditions:

The above copyright notice and this permission notice applies to all licensees
and shall be included in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF TITLE, MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
BE INCORPORATED BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF, OR IN CONNECTION
WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

Except as contained in this notice, the name of Be Incorporated shall not be
used in advertising or otherwise to promote the sale, use or other dealings in
this Software without prior written authorization from Be Incorporated.

Tracker(TM), Be(R), BeOS(R), and BeIA(TM) are trademarks or registered trademarks
of Be Incorporated in the United States and other countries. Other brand product
names are registered trademarks or trademarks of their respective holders.
All rights reserved.
*/

//	DesktopPoseView adds support for displaying integrated desktops
//	from multiple volumes to BPoseView
//
//	Used by the Desktop window and by the root view in file panels

#include <NodeMonitor.h>
#include <Path.h>
#include <VolumeRoster.h>
#include <Volume.h>

#include "DesktopPoseView.h"
#include "TFSContext.h"
#include "Tracker.h"


DesktopPoseView::DesktopPoseView(Model *model, BRect frame, uint32 viewMode,
	uint32 resizeMask)
	:	BPoseView(model, frame, viewMode, resizeMask)
{
}

EntryListBase *
DesktopPoseView::InitDesktopDirentIterator(BPoseView *nodeMonitoringTarget,
	const entry_ref *ref)
{
	// the desktop dirent iterator knows how to iterate over all the volumes,
	// integrated onto the desktop

	Model sourceModel(ref, false, true);
	if (sourceModel.InitCheck() != B_OK) 
		return 0;

	CachedEntryIteratorList *result = new DesktopEntryListCollection();
	
	ASSERT(!sourceModel.IsQuery());
	ASSERT(sourceModel.Node());
	BDirectory *sourceDirectory = dynamic_cast<BDirectory *>(sourceModel.Node());

	dev_t sourceDevice = sourceModel.NodeRef()->device;
	
	ASSERT(sourceDirectory);
	
	// build an iterator list, start with boot
	EntryListBase *perDesktopIterator = new CachedDirectoryEntryList(
		*sourceDirectory);

	result->AddItem(perDesktopIterator);
	if (nodeMonitoringTarget)
		TTracker::WatchNode(sourceModel.NodeRef(), B_WATCH_DIRECTORY
			| B_WATCH_NAME | B_WATCH_STAT | B_WATCH_ATTR, nodeMonitoringTarget);


	// add the other volumes
	BVolumeRoster roster;
	roster.Rewind();
	BVolume volume;
	while (roster.GetNextVolume(&volume) == B_OK) {

		if (volume.Device() == sourceDevice)
			// got that already
			continue;

		if (!volume.IsPersistent())
			continue;

		if (!DesktopPoseView::ShouldIntegrateVolume(&volume))
			continue;

		BDirectory root;
		if (volume.GetRootDirectory(&root) == B_OK) {
			
			BDirectory remoteDesktop;

			TFSContext::GetDesktopDir(remoteDesktop, volume.Device());
			perDesktopIterator = new CachedDirectoryEntryList(remoteDesktop);

			result->AddItem(perDesktopIterator);

			node_ref nodeRef;
			remoteDesktop.GetNodeRef(&nodeRef);

			if (nodeMonitoringTarget)
				TTracker::WatchNode(&nodeRef, B_WATCH_DIRECTORY
					| B_WATCH_NAME | B_WATCH_STAT | B_WATCH_ATTR,
					nodeMonitoringTarget);
		}
	}

	
	if (result->Rewind() != B_OK) {
		delete result;
		if (nodeMonitoringTarget)
			nodeMonitoringTarget->HideBarberPole();
		return NULL;
	}

	return result;
}


EntryListBase *
DesktopPoseView::InitDirentIterator(const entry_ref *ref)
{
	return InitDesktopDirentIterator(this, ref);
}

bool 
DesktopPoseView::FSNotification(const BMessage *message)
{
	dev_t device;

	switch (message->FindInt32("opcode")) {
		case B_DEVICE_MOUNTED:
			{
				if (message->FindInt32("new device", &device) != B_OK)
					break;
	
				ASSERT(TargetModel());
				if (TTracker::MountVolumesOntoDesktop()) {
					// place an icon for the volume onto the desktop
					BVolume volume(device);
					if (volume.InitCheck() == B_OK
						&& !volume.IsShared() || TTracker::MountSharedVolumesOntoDesktop())
						CreateVolumePose(&volume, true);
				}

				if (!TTracker::IntegrateNonBootBeOSDesktops())
					break;

				BDirectory remoteDesktop;
				BEntry entry;
				BVolume volume(device);

				if (ShouldIntegrateVolume(&volume)
					&& TFSContext::GetDesktopDir(remoteDesktop, volume.Device()) == B_OK
					&& remoteDesktop.GetEntry(&entry) == B_OK) {
					// place desktop items from the mounted volume onto the desktop
					Model model(&entry);
					if (model.InitCheck() == B_OK) 
						AddPoses(&model);
				}
			}
			break;
	}
	return _inherited::FSNotification(message);
}

namespace BPrivate {

bool
ShouldShowDesktopPose(dev_t device, const Model *model, const PoseInfo *)
{
	if (model->NodeRef()->device != device) {
		// avoid having more than one Trash
		BDirectory remoteTrash;
		if (TFSContext::GetTrashDir(remoteTrash, model->NodeRef()->device) == B_OK) {
			node_ref remoteTrashNodeRef;
			remoteTrash.GetNodeRef(&remoteTrashNodeRef);
			if (remoteTrashNodeRef == *model->NodeRef())
				return false;
		}
	}
	return true;
}

}

bool
DesktopPoseView::ShouldShowPose(const Model *model, const PoseInfo *poseInfo)
{
	ASSERT(TargetModel());
	if (!ShouldShowDesktopPose(TargetModel()->NodeRef()->device, model, poseInfo))
		return false;

	return _inherited::ShouldShowPose(model, poseInfo);
}

bool
DesktopPoseView::Represents(const node_ref *ref) const
{
	//	When the Tracker is set up to integrate non-boot beos volumes,
	//	it represents the home/Desktop folders of all beos volumes

	if (TTracker::IntegrateNonBootBeOSDesktops()) {
		BDirectory deviceDesktop;
		TFSContext::GetDesktopDir(deviceDesktop, ref->device);
		node_ref nref;
		deviceDesktop.GetNodeRef(&nref);
		return nref == *ref;
	}
		
	
	return _inherited::Represents(ref);
}

bool
DesktopPoseView::Represents(const entry_ref *ref) const
{
	BEntry entry(ref);
	node_ref nref;
	entry.GetNodeRef(&nref);
	return Represents(&nref);
}

DesktopEntryListCollection::DesktopEntryListCollection()
{
}

