
#include "TFSContext.h"
#include "Attributes.h"
#include "Commands.h"
#include "Tracker.h"
#include "ThreadMagic.h"
#include "Bitmaps.h"
#include "StatusWindow.h"
#include "OverrideAlert.h"
#include "FSDialogWindow.h"

#include <Application.h>
#include <Autolock.h>
#include <fs_attr.h>
#include <Roster.h>
#include <Alert.h>
#include <Beep.h>
#include <errno.h>
#include <math.h>

// Various info
//	- ContainerView does pulse managemant on it's own by a BMessageRunner. It calls StatusView::Pulse which
//	  calls ContainerView::Pulse. This is required to be independent from the holding window, that might
//    be the desktop window also.

// Current limitations
//	- Cancels all operations in case of an error or interaction request. (beep()s also)
//	- Async mode does not yet work due to FunctionObject problems
//	- Does not yet checks for well known entries
//	- Estimated time is redrawn when window is resized (Draw() seems to be called with Bounds() instead of the rect to be updated)
//	- AlertPosition seems to be bad (I guess it's my fault, but what the...)
//	- See list in the beginning of FSContext.cpp

// TODO
//	- Be smart and auto-pause operations that would be involved in a big headbanging party
//  - Draw a string that tells what will happen by pressinf skip (skip file or skip dir)
//  - Make sure button size is a multiple of 2 (looks better with small buttons)
//  - kProgressViewXOffsetFromRight replaced with dynamic calc.
//  - Cut off StatusWindow code from TFSContext
//  - Reenable well known entry check

// BUGS
//  - Crash at exit... some memory clobber is reported, but the information is useless
//  - StatusView separator sometimes drawn for a single view, too.
//  - StatusView redraw at the right, below the buttons is bad on heavy system load. Dunno why.

#define		FS_PRINT_ESTIMATION_INFO	0

namespace BPrivate {

using namespace fs;

WellKnowEntryList *	WellKnowEntryList :: self = 0;

void
TFSContext :: initialize() {
	mDialogWindow	= 0;
	
	mConnectedToStatusWindow	= true;
	mInteractive				= true;
	mProgressInfoEnabled		= true;
	mOperationBegun				= false;
	mWasOverheadSWRunning		= false;
	mOperationStringDirty		= true;	

	mPause				= false;
	mAutoPaused			= false;
	mShouldAutoPause	= true;
	mCancel				= false;
	mSkipOperation		= false;
	mSkipDirectory		= false;
	mSkipEntry			= false;
	
	mWorkingThread = 0;
	mPreviousTimeGuess = 0;
}

TFSContext :: TFSContext() :
				mElapsedStopWatch("", true),
				mOverheadStopWatch("", true),
				mEntryIterator(mEntryList) {

	initialize();
}

// these ctors collect a list of entries from the given parameters
TFSContext :: TFSContext(const PoseList &inlist) :
				mElapsedStopWatch("", true),
				mOverheadStopWatch("", true),
				mEntryList(inlist),
				mEntryIterator(mEntryList) {

	initialize();
}

TFSContext :: TFSContext(const BMessage &in_msg) :
				mElapsedStopWatch("", true),
				mOverheadStopWatch("", true),
				mEntryList(in_msg),
				mEntryIterator(mEntryList) {

	initialize();
}

TFSContext :: TFSContext(const entry_ref &in_ref) :
				mElapsedStopWatch("", true),
				mOverheadStopWatch("", true),
				mEntryList(in_ref),
				mEntryIterator(mEntryList) {
				
	initialize();
}

TFSContext :: ~TFSContext() {
	if (IsConnectedToStatusWindow())
		gStatusWindow().Remove(*this);
		
	if (mDialogWindow  &&  mDialogWindow -> Lock())
		mDialogWindow -> Quit();
}


status_t
TFSContext :: CalculateItemsAndSize(ProgressInfo &ininfo) {
	CommonFunctionBeforeOperations();
	return inherited::CalculateItemsAndSize(mEntryIterator, ininfo);	// XXX without inherited gcc can't bind the inherited function ?!
}

status_t
TFSContext :: CopyTo(BDirectory &target_dir, bool async) {
	CommonFunctionBeforeOperations();
	return async ?
		inherited::CopyTo(NewIterator(), target_dir, true) :
		inherited::CopyTo(mEntryIterator, target_dir);
}

status_t
TFSContext :: Duplicate(bool async) {
	CommonFunctionBeforeOperations();
	return async ?
		inherited::Duplicate(NewIterator(), true) :
		inherited::Duplicate(mEntryIterator);
}

status_t
TFSContext :: CreateLinkTo(BDirectory &target_dir, bool relative, bool async) {
	CommonFunctionBeforeOperations();
	return async ?
		inherited::CreateLinkTo(NewIterator(), target_dir, relative, true) :
		inherited::CreateLinkTo(mEntryIterator, target_dir, relative);
}

status_t
TFSContext :: MoveTo(BDirectory &target_dir, bool async) {
	CommonFunctionBeforeOperations();
	return async ?
		inherited::MoveTo(NewIterator(), target_dir, true) :
		inherited::MoveTo(mEntryIterator, target_dir);
}

status_t
TFSContext :: MoveToTrash(bool async) {
	CommonFunctionBeforeOperations();
	return async ?
		inherited::MoveToTrashAsync(NewIterator(), true) :
		inherited::MoveToTrash(mEntryIterator);
}

status_t
TFSContext :: RestoreFromTrash(bool async) {
	CommonFunctionBeforeOperations();
	return async ?
		inherited::RestoreFromTrash(NewIterator(), true) :
		inherited::RestoreFromTrash(mEntryIterator);
}

status_t
TFSContext :: Remove(bool in_ask_user, bool async) {
	CommonFunctionBeforeOperations();
	return async ?
		inherited::Remove(NewIterator(), in_ask_user, true) :
		inherited::Remove(mEntryIterator, in_ask_user);
}

void
TFSContext :: CommonFunctionBeforeOperations() {
	mEntryIterator.Rewind();
	mPointListPos = mPointList.begin();
	mStartTime = system_time();
}

void
TFSContext :: OperationBegins() {
	mElapsedStopWatch.Reset();
	mOverheadStopWatch.Reset();
	mOperationBegun = true;
}

int32
TFSContext :: EstimatedTimeLeft() {

	static const int32 kMinimalEntryCountForEstimation = 20;

	if (mProgressInfo.IsTotalEnabled() == false  ||  DidOperationBegin() == false  ||
		mElapsedStopWatch.ElapsedTime() < 2000000)
		return 0;

	ASSERT(mProgressInfo.EntryProgress() <= 1		&&  mProgressInfo.EntryProgress() >= 0);
	ASSERT(mProgressInfo.TotalSizeProgress() <= 1	&&  mProgressInfo.TotalSizeProgress() >= 0);

	float max		= (float)(mProgressInfo.mTotalEntryCount * 2);

	float current1	= (float)mProgressInfo.mCurrentEntryCount;
	float weight1 = mProgressInfo.TotalSizeProgress();

	float current2;
	float weight2;

	if (mProgressInfo.IsTotalSizeProgressEnabled()) {
	
		current2 = (float)mProgressInfo.mCurrentSize * mProgressInfo.mTotalEntryCount / mProgressInfo.mTotalSize;
		weight1 = mProgressInfo.TotalSizeProgress();
		weight2 = mProgressInfo.EntryProgress();
		float f = (weight1 + weight2) / 2;
		weight1 /= f;
		weight2 /= f;
	
		weight1 = powf(weight1, 0.4);
		weight2 = powf(weight2, 0.4);
	
		f = (weight1 + weight2) / 2;
		weight1 /= f;
		weight2 /= f;
	
		if (mProgressInfo.mTotalEntryCount < kMinimalEntryCountForEstimation) {	// to few files, don't bother
			current1 = 0;
			weight1 = 0;
			weight2 = 2;
		}
	} else {
		current2 = 0;
		weight2 = 0;
		weight1 = 2;
	}
	
	float time = mElapsedStopWatch.ElapsedTime() / 1000000;
	float guess = time / (current1 * weight1 + current2 * weight2) * max - time;

#if FS_PRINT_ESTIMATION_INFO
	printf("elaps: %.1f,\toverh: %.1f,\tprogr: %.2f, %.2f,\tguess: %.1f * %.1f + %.1f * %.1f  = %.1f/%.1f -> \t%.2f\n",
			(float)(mElapsedStopWatch.ElapsedTime() / 1000000),
			(float)(mOverheadStopWatch.ElapsedTime() / 1000000),
			mProgressInfo.EntryProgress(),
			mProgressInfo.TotalSizeProgress(),
			current1, weight1, current2, weight2,
			current1 * weight1 + current2 * weight2, max, guess);
#endif		

	if (guess > 10000  ||  guess < 0)
		return 0;							// no idea

	if (mPreviousTimeGuess == 0)
		mPreviousTimeGuess = guess;
		
	time = (mPreviousTimeGuess + guess) / 2;
	mPreviousTimeGuess = guess;
	
	return (int32)time;
}

bool
TFSContext :: Pause() {

	mShouldAutoPause = false;
	mAutoPaused = false;
	
	if (mPause) {
	
		HardResume();
	} else {
	
		mPause = true;
	}
	
	mOperationStringDirty = true;
	return mPause;
}

void
TFSContext :: SoftResume() {

	ReallocateBuffer();
	
	mElapsedStopWatch.Resume();
	if (IsEffectiveFileCopyRunning() == false)
		mOverheadStopWatch.Resume();

	if (mWorkingThread != 0)
		resume_thread(mWorkingThread);

	mOperationStringDirty = true;
}

void
TFSContext :: CheckCancel() {

	if (mCancel) {
	
		mCancel = false;
		FS_CONTROL_THROW(kCancel);
	}
	
	if (mSkipOperation) {
		mSkipOperation = false;
		
		if (IsAnswerPossible(fSkipOperation)) {
			FS_CONTROL_THROW(kSkipOperation);
		}
	}
	
	if (mSkipDirectory) {
		mSkipDirectory = false;
		
		if (IsAnswerPossible(fSkipDirectory)) {
			FS_CONTROL_THROW(kSkipDirectory);
		}
	}

	if (mSkipEntry) {
		mSkipEntry = false;
		
		if (IsAnswerPossible(fSkipEntry)) {
			FS_CONTROL_THROW(kSkipEntry);
		}
	}

	if (mPause) {
	
		mElapsedStopWatch.Suspend();
		mOverheadStopWatch.Suspend();

		ReleaseBuffer();

		mWorkingThread = find_thread(0);
		suspend_thread(mWorkingThread);		// StatusView will call Pause() that will wake the thread up if required

		CheckCancel();						// check again to quickly catch a cancel or skipop if it was blocked by a pause
	} else if (mShouldAutoPause  &&  mAutoPaused == false) {
		// should not check again for autopause if it was resumed once
		
		if (CurrentOperation() == kCalculateItemsAndSize  ||  RootOperation() != kCopy) {
			if (mProgressInfo.mTotalEntryCount <= 10)
				return;
		}
		
		if (gStatusWindow().ShouldPause(*this)) {
			mPause = true;
			mAutoPaused = true;
			mOperationStringDirty = true;
			
			CheckCancel();		// XXX maybe not usefull here?
		}
	}
}


status_t
TFSContext :: SetPoseLocation(ino_t in_dest_dir_inode, BNode &in_dest_node, const BPoint &in_point) {

	PoseInfo poseInfo;
	poseInfo.fInvisible = false;
	poseInfo.fInitedDirectory = in_dest_dir_inode;
	poseInfo.fLocation = in_point;

	status_t rc = in_dest_node.WriteAttr(kAttrPoseInfo, B_RAW_TYPE, 0, &poseInfo, sizeof(poseInfo));

	if (rc == sizeof(poseInfo))
		return B_OK;
	
	return rc;
}

void
TFSContext :: InitProgressIndicator() {
	if (mProgressInfoEnabled) {
		mConnectedToStatusWindow = true;
		gStatusWindow().Add(*this);
	}
	
//	snooze(50000000);	// XXX
}


bool
TFSContext :: ErrorHandler(status_t in_err) {

	if (mCancel  ||  IsInteractive() == false) {
		FS_CONTROL_THROW(kCancel);
	}

	PauseStopWatch				p1(mElapsedStopWatch);
	PauseStopWatch_ResumeIf 	p2(mOverheadStopWatch, this, &TFSContext::IsEffectiveFileCopyRunning);
	
	command cmd = DefaultErrorAnswer(in_err);

	if (cmd != 0) {

		if		(cmd == kRetryOperation)	return true;
		else if	(cmd == kIgnore)			return false;

		TRESPASS();
		beep();
		FS_CONTROL_THROW(kCancel);
	}

	bool def;

	try {
	
		mDialogWindow = new FSDialogWindow(&cmd, &def, &mDialogWindow, *this, in_err);
		mDialogWindow -> Go();
	
		do {
		
			mPause = true;
			CheckCancel();			// fall asleep until we are resumed by the dialog window
			
		} while (mDialogWindow != 0);
		
	} catch (FSException e) {
	
		if (mDialogWindow != 0  &&  mDialogWindow -> Lock()) {
			mDialogWindow -> Quit();
			mDialogWindow = 0;
		}
		throw;
	}

	if (def  &&  IsDefaultableCommand(cmd))
		SetDefaultErrorAnswer(cmd, in_err);

	if (IsThrowableCommand(cmd)) {
		FS_CONTROL_THROW(cmd);
	}
	
	if (cmd == kRetryOperation)
		return true;
	else if (cmd == kIgnore)
		return false;
	else
		TRESPASS();
	
	return true;
}

FSContext::command
TFSContext :: Interaction(interaction in_code, char *source_name, char *target_name) {

	if (mCancel  ||  IsInteractive() == false) {
		FS_CONTROL_THROW(kCancel);
	}

	command cmd = DefaultInteractionAnswer(in_code);
	if (cmd != 0)
		return cmd;

	PauseStopWatch				p1(mElapsedStopWatch);
	PauseStopWatch_ResumeIf 	p2(mOverheadStopWatch, this, &TFSContext::IsEffectiveFileCopyRunning);

	bool def;
	try {

		mDialogWindow = new FSDialogWindow(&cmd, &def, &mDialogWindow, source_name,
											target_name, *this, in_code);
		mDialogWindow -> Go();
	
		do {
		
			mPause = true;
			CheckCancel();			// fall asleep until we are resumed by the dialog window
			
		} while (mDialogWindow != 0);

	} catch (FSException e) {
	
		if (mDialogWindow != 0  &&  mDialogWindow -> Lock()) {
			mDialogWindow -> Quit();
			mDialogWindow = 0;
		}
		throw;
	}
	
	if (def  &&  IsDefaultableCommand(cmd))
		SetDefaultInteractionAnswer(in_code, cmd);
	
	if (IsThrowableCommand(cmd)) {
		FS_CONTROL_THROW(cmd);
	}
	
	return cmd;
}

BWindow	*
TFSContext :: DialogWindow() const {
	return mDialogWindow;
}

void
TFSContext :: NextEntryCreated(node_ref &in_dir_ref, BNode &in_node) {

	if (mPointListPos < mPointList.end()) {		// if we have position info then write it out
		PoseInfo pose_info;
		
		pose_info.fInvisible = false;
		pose_info.fInitedDirectory = in_dir_ref.node;
		pose_info.fLocation = *mPointListPos;
		
		in_node.WriteAttr(kAttrPoseInfo, B_RAW_TYPE, 0, &pose_info, sizeof(PoseInfo));
		
		++mPointListPos;
	}
}

void
TFSContext :: DirectoryTrashed(node_ref &in_node) {

	BMessage message(kCloseWindowAndChildren);
	message.AddData("node_ref", B_RAW_TYPE, &in_node, sizeof(node_ref));
	be_app->PostMessage(&message);
}

void
TFSContext :: AboutToDeleteOrTrash(BEntry &entry) {

	if (entry.IsDirectory()) {							// unmount if volume
		BDirectory dir;
		
		status_t rc;
		FS_OPERATION(dir.SetTo(&entry));

		if (dir.IsRootDirectory()) {
			struct stat st;
			FS_OPERATION(entry.GetStat(&st));
			
			BVolume	boot;
			FS_OPERATION(BVolumeRoster().GetBootVolume(&boot));
			
			if (boot.Device() != st.st_dev) {
				
				BMessage message(kUnmountVolume);
				message.AddInt32("device_id", st.st_dev);
				be_app->PostMessage(&message);

				if (IsAnswerPossible(fSkipEntry)) {
					FS_CONTROL_THROW(kSkipEntry);
				} else {
					return;
				}
			} else {
				for (;;)
					Interaction(kCannotUnmountBootVolume);
			}
		}
	}
}


status_t 
TFSContext :: GetTrackerSettingsDir(BPath &path, bool autoCreate) {

	status_t rc = find_directory(B_USER_SETTINGS_DIRECTORY, &path, autoCreate);
	if (rc != B_OK)
		return rc;
		
	path.Append("Tracker");

	return mkdir(path.Path(), 0777) ? B_OK : errno;
}


void
TFSContext :: CreateSpecialDirs() {

	BVolume volume;
	BVolumeRoster roster;

	while (roster.GetNextVolume(&volume) == B_OK) {
		
		if (volume.IsReadOnly() || !volume.IsPersistent())
			continue;
		
		BPath path;
		find_directory(B_DESKTOP_DIRECTORY, &path, true, &volume);
		find_directory(B_TRASH_DIRECTORY, &path, true, &volume);

		BDirectory trashDir;
		if (GetTrashDir(trashDir, volume.Device()) == B_OK) {
			size_t size;
			const void* data;
			if ((data = GetTrackerResources()->LoadResource('ICON', kResTrashIcon, &size)) != 0)
				trashDir.WriteAttr(kAttrLargeIcon, B_COLOR_8_BIT_TYPE, 0, data, size);

			if ((data = GetTrackerResources()->LoadResource('MICN', kResTrashIcon, &size)) != 0)
				trashDir.WriteAttr(kAttrMiniIcon, B_COLOR_8_BIT_TYPE, 0, data, size);
		}
	}
}


status_t
TFSContext :: SetPoseLocation(BEntry &entry, const BPoint &point) {

	status_t rc;

	BNode node(&entry);
	FS_STATIC_OPERATION(node.InitCheck());
	
	BDirectory parent;
	FS_STATIC_OPERATION(entry.GetParent(&parent));
	
	node_ref destNodeRef;
	FS_STATIC_OPERATION(parent.GetNodeRef(&destNodeRef));
	
	return SetPoseLocation(destNodeRef.node, node, point);
}

bool
TFSContext :: GetPoseLocation(const BNode &node, BPoint &point) {

	PoseInfo poseInfo;
	if (ReadAttr(node, kAttrPoseInfo, kAttrPoseInfoForeign,
			B_RAW_TYPE, 0, &poseInfo, sizeof(poseInfo), &PoseInfo::EndianSwap)
			== kReadAttrFailed)
		return false;
	
	if (poseInfo.fInitedDirectory == -1LL)
		return false;

	point = poseInfo.fLocation;

	return true;
}







// Stuffs remained from the old FSUtils

const char *kFindAlternativeStr = "Would you like to find some other suitable application?";
const char *kFindApplicationStr = "Would you like to find a suitable application to "
	"open the file?";


ReadAttrResult
ReadAttr(const BNode &node, const char *hostAttrName, const char *foreignAttrName,
	type_code type, off_t offset, void *buffer, size_t length,
	void (*swapFunc)(void *), bool isForeign) {
	
	if (!isForeign && node.ReadAttr(hostAttrName, type, offset, buffer, length) == (ssize_t)length)
		return kReadAttrNativeOK;

	// PRINT(("trying %s\n", foreignAttrName));
	// try the other endianness	
	if (node.ReadAttr(foreignAttrName, type, offset, buffer, length) != (ssize_t)length)
		return kReadAttrFailed;
	
	// PRINT(("got %s\n", foreignAttrName));
	if (swapFunc)
		(swapFunc)(buffer);		// run the endian swapper

	return kReadAttrForeignOK;
}

ReadAttrResult 
GetAttrInfo(const BNode &node, const char *hostAttrName, const char *foreignAttrName,
	type_code *type , size_t *size) {

	attr_info info;
	
	if (node.GetAttrInfo(hostAttrName, &info) == B_OK) {
		if (type)
			*type = info.type;
		if (size)
			*size = (size_t)info.size;

		return kReadAttrNativeOK;
	}

	if (node.GetAttrInfo(foreignAttrName, &info) == B_OK) {
		if (type)
			*type = info.type;
		if (size)
			*size = (size_t)info.size;

		return kReadAttrForeignOK;
	}
	return kReadAttrFailed;
}

// launching code

static status_t
TrackerOpenWith(const BMessage *refs)
{
	BMessage clone(*refs);
	ASSERT(dynamic_cast<TTracker *>(be_app));
	ASSERT(clone.what);
	clone.AddInt32("launchUsingSelector", 0);
	// runs the Open With window
	be_app->PostMessage(&clone);

	return B_OK;
}

static void 
AsynchLaunchBinder(void (*func)(const entry_ref &, const BMessage &, bool on),
	const entry_ref *entry, const BMessage *message, bool on)
{
	entry_ref fake_entry;
	BMessage fake_msg;

	LaunchInNewThread("LaunchTask", B_NORMAL_PRIORITY, func,
		(entry) ? *entry : fake_entry, (message) ? *message : fake_msg, on);
}

static bool
SniffIfGeneric(const entry_ref *ref)
{
	BNode node(ref);
	char type[B_MIME_TYPE_LENGTH];
	BNodeInfo info(&node);
	if (info.GetType(type) == B_OK && strcasecmp(type, B_FILE_MIME_TYPE) != 0)
		// already has a type and it's not octet stream
		return false;
	
	BPath path(ref);
	if (path.Path()) {
		// force a mimeset
		node.RemoveAttr(kAttrMIMEType);
		update_mime_info(path.Path(), 0, 1, 1);
	}
	
	return true;
}

static void
SniffIfGeneric(const BMessage *refs)
{
	entry_ref ref;
	for (int32 index = 0; ; index++) {
		if (refs->FindRef("refs", index, &ref) != B_OK)
			break;
		SniffIfGeneric(&ref);
	}
}

static void
_TrackerLaunchAppWithDocuments(const entry_ref &appRef, const BMessage &refs, bool openWithOK)
{
	team_id team;

	status_t error = B_ERROR;
	BString alertString;

	for (int32 mimesetIt = 0; ; mimesetIt++) {
		error = be_roster->Launch(&appRef, &refs, &team);
		if (error == B_ALREADY_RUNNING)
			// app already running, not really an error
			error = B_OK;
		
		if (error == B_OK)
			break;

		if (mimesetIt > 0)
			break;
		
		// failed to open, try mimesetting the refs and launching again
		SniffIfGeneric(&refs);
	}
	
	if (error == B_OK) {
		// close possible parent window, if specified
		node_ref *nodeToClose = 0;
		int32 numBytes;
		refs.FindData("nodeRefsToClose", B_RAW_TYPE, (const void **)&nodeToClose, &numBytes);
		if (nodeToClose)
			dynamic_cast<TTracker *>(be_app)->CloseParent(*nodeToClose);
	} else {
		alertString << "Could not open \"" << appRef.name << "\" (" << strerror(error) << "). ";
		if (refs.what != 12345678 && openWithOK) {
			alertString << kFindAlternativeStr;
			if ((new BAlert("", alertString.String(), "Cancel", "Find", 0,
					B_WIDTH_AS_USUAL, B_WARNING_ALERT))->Go() == 1)
				error = TrackerOpenWith(&refs);
		} else
			(new BAlert("", alertString.String(), "Cancel", 0, 0,
				B_WIDTH_AS_USUAL, B_WARNING_ALERT))->Go();
	}
}

extern "C" char** environ;
extern "C" _IMPEXP_ROOT status_t _kload_image_etc_(int argc, char **argv, char **envp,
	char *buf, int bufsize);


static status_t
LoaderErrorDetails(const entry_ref *app, BString &details)
{
	BPath path;
	BEntry appEntry(app, true);
	status_t result = appEntry.GetPath(&path);
	
	if (result != B_OK)
		return result;
	
	char *argv[2] = { const_cast<char *>(path.Path()), 0};

	result = _kload_image_etc_(1, argv, environ, details.LockBuffer(1024), 1024);
	details.UnlockBuffer();

	return B_OK;
}

static void
_TrackerLaunchDocuments(const entry_ref &/*doNotUse*/, const BMessage &refs,
	bool openWithOK)
{
	BMessage copyOfRefs(refs);
	
	entry_ref documentRef;
	if (copyOfRefs.FindRef("refs", &documentRef) != B_OK)
		// nothing to launch, we are done
		return;

	status_t error = B_ERROR;
	entry_ref app;
	BMessage *refsToPass = NULL;
	BString alertString;
	const char *alternative = 0;

	for (int32 mimesetIt = 0; ; mimesetIt++) {
		alertString = "";
		error = be_roster->FindApp(&documentRef, &app);
			
		if (error != B_OK && mimesetIt == 0) {
			SniffIfGeneric(&copyOfRefs);
			continue;
		}
	
		
		if (error != B_OK) {
			alertString << "Could not find an application to open \"" << documentRef.name
				<< "\" (" << strerror(error) << "). ";
			if (openWithOK)
				alternative = kFindApplicationStr;

			break;
		} else {
	
			BEntry appEntry(&app, true);
			for (int32 index = 0;;) {
				// remove the app itself from the refs received so we don't try
				// to open ourselves
				entry_ref ref;
				if (copyOfRefs.FindRef("refs", index, &ref) != B_OK)
					break;
				
				// deal with symlinks properly
				BEntry documentEntry(&ref, true);
				if (appEntry == documentEntry) {
					PRINT(("stripping %s, app %s \n", ref.name, app.name));
					copyOfRefs.RemoveData("refs", index);
				} else {
					PRINT(("leaving %s, app %s  \n", ref.name, app.name));
					index++;
				}
			}
	
			refsToPass = CountRefs(&copyOfRefs) > 0 ? &copyOfRefs: 0;
			team_id team;
			error = be_roster->Launch(&app, refsToPass, &team);
			if (error == B_ALREADY_RUNNING) 
				// app already running, not really an error
				error = B_OK;
			if (error == B_OK || mimesetIt != 0)
				break;
			
			SniffIfGeneric(&copyOfRefs);
		}
	}
	
	if (error != B_OK && alertString.Length() == 0) {
		BString loaderErrorString;
		bool openedDocuments = true;
	
		if (!refsToPass) {
			// we just double clicked the app itself, do not offer to
			// find a handling app
			openWithOK = false;
			openedDocuments = false;
		}

		if (error == B_LAUNCH_FAILED_EXECUTABLE && !refsToPass) {
			alertString << "Could not open \"" << app.name
				<< "\". The file is mistakenly marked as executable. ";
			alternative = kFindApplicationStr;
		} else if (error == B_LAUNCH_FAILED_APP_IN_TRASH) {
			alertString << "Could not open \"" << documentRef.name
				<< "\" because application \"" << app.name << "\" is in the trash. ";
			alternative = kFindAlternativeStr;
		} else if (error == B_LAUNCH_FAILED_APP_NOT_FOUND) {
			alertString << "Could not open \"" << documentRef.name << "\" "
				<< "(" << strerror(error) << "). ";
			alternative = kFindAlternativeStr;
		} else if (error == B_MISSING_SYMBOL
			&& LoaderErrorDetails(&app, loaderErrorString) == B_OK) {
			alertString << "Could not open \"" << documentRef.name << "\" ";
			if (openedDocuments)
				alertString << "with application \"" << app.name << "\" ";
			alertString << "(Missing symbol: " << loaderErrorString << "). \n";
			alternative = kFindAlternativeStr;
		} else if (error == B_MISSING_LIBRARY
			&& LoaderErrorDetails(&app, loaderErrorString) == B_OK) {
			alertString << "Could not open \"" << documentRef.name << "\" ";
			if (openedDocuments)
				alertString << "with application \"" << app.name << "\" ";
			alertString << "(Missing library: " << loaderErrorString << "). \n";
			alternative = kFindAlternativeStr;
		} else {
			alertString << "Could not open \"" << documentRef.name
				<< "\" with application \"" << app.name << "\" (" << strerror(error) << "). ";
			alternative = kFindAlternativeStr;
		}
	}

	if (error != B_OK) {
		if (openWithOK) {
			ASSERT(alternative);
			alertString << alternative;
			if ((new BAlert("", alertString.String(), "Cancel", "Find", 0,
					B_WIDTH_AS_USUAL, B_WARNING_ALERT))->Go() == 1)
				error = TrackerOpenWith(&refs);
		} else 
			(new BAlert("", alertString.String(), "Cancel", 0, 0,
					B_WIDTH_AS_USUAL, B_WARNING_ALERT))->Go();
	}
}

// the following three calls don't return any reasonable error codes,
// should fix that, making them void

status_t 
TrackerLaunch(const entry_ref *appRef, const BMessage *refs, bool async, bool openWithOK)
{
	if (!async)
		_TrackerLaunchAppWithDocuments(*appRef, *refs, openWithOK);
	else
		AsynchLaunchBinder(&_TrackerLaunchAppWithDocuments, appRef, refs, openWithOK);

	return B_OK;
}

status_t 
TrackerLaunch(const entry_ref *appRef, bool async)
{
	BMessage msg(12345678);
	
	if (!async)
		_TrackerLaunchAppWithDocuments(*appRef, msg, false);
	else
		AsynchLaunchBinder(&_TrackerLaunchAppWithDocuments, appRef, &msg, false);

	return B_OK;
}

status_t 
TrackerLaunch(const BMessage *refs, bool async, bool openWithOK)
{
	entry_ref fake;
	if (!async) 
		_TrackerLaunchDocuments(fake, *refs, openWithOK);
	else
		AsynchLaunchBinder(&_TrackerLaunchDocuments, &fake, refs, openWithOK);

	return B_OK;
}

status_t
LaunchBrokenLink(const char *signature, const BMessage *refs)
{
	// This call is to support a hacky workaround for double-clicking
	// broken refs for cifs
	be_roster->Launch(signature, const_cast<BMessage *>(refs));
	return B_OK;
}

// external launch calls; need to be robust, work if Tracker is not running

_IMPEXP_TRACKER status_t 
FSLaunchItem(const entry_ref *application, const BMessage *refsReceived,
	bool async, bool openWithOK)
{
	return TrackerLaunch(application, refsReceived, async, openWithOK);
}


_IMPEXP_TRACKER status_t 
FSOpenWith(BMessage *listOfRefs)
{
	status_t result = B_ERROR;
	listOfRefs->what = B_REFS_RECEIVED;
	
	if (dynamic_cast<TTracker *>(be_app)) 
		result = TrackerOpenWith(listOfRefs);
	else 
		ASSERT(!"not yet implemented");

	return result;
}

// legacy calls, need for compatibility

void 
FSOpenWithDocuments(const entry_ref *executable, BMessage *documents)
{
	TrackerLaunch(executable, documents, true);
	delete documents;
}

status_t
FSLaunchUsing(const entry_ref *ref, BMessage *listOfRefs)
{
	BMessage temp(B_REFS_RECEIVED);
	if (!listOfRefs) {
		ASSERT(ref);
		temp.AddRef("refs", ref);
		listOfRefs = &temp;
	}
	FSOpenWith(listOfRefs);
	return B_OK;
}

status_t
FSLaunchItem(const entry_ref *ref, BMessage* message, int32, bool async)
{
	if (message) 
		message->what = B_REFS_RECEIVED;

	status_t result = TrackerLaunch(ref, message, async, true);
	delete message;
	return result;
}


void
FSLaunchItem(const entry_ref *ref, BMessage *message, int32 workspace)
{
	FSLaunchItem(ref, message, workspace, true);
}



directory_which 
WellKnowEntryList::Match(const node_ref *node)
{
	const WellKnownEntry *result = MatchEntry(node);
	if (result)
		return result->which;
	
	return (directory_which)-1;
}

const WellKnowEntryList::WellKnownEntry *
WellKnowEntryList::MatchEntry(const node_ref *node)
{
	if (!self)
		self = new WellKnowEntryList();
	
	return self->MatchEntryCommon(node);
}

const WellKnowEntryList::WellKnownEntry *
WellKnowEntryList::MatchEntryCommon(const node_ref *node)
{	
	uint32 count = entries.size();
	for (uint32 index = 0; index < count; index++)
		if (*node == entries[index].node)
			return &entries[index];
	
	return NULL;
}


void
WellKnowEntryList::Quit()
{
	delete self;
	self = NULL;
}

void
WellKnowEntryList::AddOne(directory_which which, const char *name)
{
	BPath path;
	if (find_directory(which, &path, true) != B_OK)
		return;
	
	BEntry entry(path.Path());
	node_ref node;
	if (entry.GetNodeRef(&node) != B_OK)
		return;
	
	entries.push_back(WellKnownEntry(&node, which, name));
}

void 
WellKnowEntryList::AddOne(directory_which which, directory_which base,
	const char *extra, const char *name)
{
	BPath path;
	if (find_directory(base, &path, true) != B_OK)
		return;
	
	path.Append(extra);
	BEntry entry(path.Path());
	node_ref node;
	if (entry.GetNodeRef(&node) != B_OK)
		return;
	
	entries.push_back(WellKnownEntry(&node, which, name));
}

void 
WellKnowEntryList::AddOne(directory_which which, const char *path, const char *name)
{
	BEntry entry(path);
	node_ref node;
	if (entry.GetNodeRef(&node) != B_OK)
		return;
	
	entries.push_back(WellKnownEntry(&node, which, name));
}


WellKnowEntryList::WellKnowEntryList()
{
	AddOne(B_BEOS_DIRECTORY, "beos");
	AddOne((directory_which)B_BOOT_DISK, "/boot", "boot");
	AddOne(B_USER_DIRECTORY, "home");
	AddOne(B_BEOS_SYSTEM_DIRECTORY, "system");
	
	AddOne(B_BEOS_FONTS_DIRECTORY, "fonts");
	AddOne(B_COMMON_FONTS_DIRECTORY, "fonts");
	AddOne(B_USER_FONTS_DIRECTORY, "fonts");
	
	AddOne(B_BEOS_APPS_DIRECTORY, "apps");
	AddOne(B_APPS_DIRECTORY, "apps");
	AddOne((directory_which)B_USER_DESKBAR_APPS_DIRECTORY, B_USER_DESKBAR_DIRECTORY,
		"Applications", "apps");

	AddOne(B_BEOS_PREFERENCES_DIRECTORY, "preferences");
	AddOne(B_PREFERENCES_DIRECTORY, "preferences");
	AddOne((directory_which)B_USER_DESKBAR_PREFERENCES_DIRECTORY, B_USER_DESKBAR_DIRECTORY,
		"Preferences", "preferences");

	AddOne((directory_which)B_USER_MAIL_DIRECTORY, B_USER_DIRECTORY, "mail", "mail");
	
	AddOne((directory_which)B_USER_QUERIES_DIRECTORY, B_USER_DIRECTORY, "queries", "queries");


	
	AddOne(B_COMMON_DEVELOP_DIRECTORY, "develop");
	AddOne((directory_which)B_USER_DESKBAR_DEVELOP_DIRECTORY, B_USER_DESKBAR_DIRECTORY,
		"Development", "develop");
	
	AddOne(B_USER_CONFIG_DIRECTORY, "config");
	
	AddOne((directory_which)B_USER_PEOPLE_DIRECTORY, B_USER_DIRECTORY, "people", "people");
	
	AddOne((directory_which)B_USER_DOWNLOADS_DIRECTORY, B_USER_DIRECTORY, "Downloads",
		"Downloads");
}


bool 
DirectoryMatchesOrContains(const BEntry *entry, directory_which which)
{
	BPath path;
	if (find_directory(which, &path, false, NULL) != B_OK)
		return false;
	
	BEntry dirEntry(path.Path());
	if (dirEntry.InitCheck() != B_OK)
		return false;
	
	if (dirEntry == *entry)
		// root level match
		return true;
	
	BDirectory dir(&dirEntry);
	return dir.Contains(entry);
}

bool 
DirectoryMatchesOrContains(const BEntry *entry, const char *additionalPath,
	directory_which which)
{
	BPath path;
	if (find_directory(which, &path, false, NULL) != B_OK)
		return false;
	
	path.Append(additionalPath);
	BEntry dirEntry(path.Path());
	if (dirEntry.InitCheck() != B_OK)
		return false;
	
	if (dirEntry == *entry)
		// root level match
		return true;
	
	BDirectory dir(&dirEntry);
	return dir.Contains(entry);
}


bool 
DirectoryMatches(const BEntry *entry, directory_which which)
{
	BPath path;
	if (find_directory(which, &path, false, NULL) != B_OK)
		return false;
	
	BEntry dirEntry(path.Path());
	if (dirEntry.InitCheck() != B_OK)
		return false;
	
	return dirEntry == *entry;
}

bool 
DirectoryMatches(const BEntry *entry, const char *additionalPath, directory_which which)
{
	BPath path;
	if (find_directory(which, &path, false, NULL) != B_OK)
		return false;
	
	path.Append(additionalPath);
	BEntry dirEntry(path.Path());
	if (dirEntry.InitCheck() != B_OK)
		return false;
	
	return dirEntry == *entry;
}


enum {
	kNotConfirmed,
	kConfirmedHomeMove,
	kConfirmedAll
};

bool
ConfirmChangeIfWellKnownDirectory(const BEntry *entry, const char *action,
	bool dontAsk, int32 *confirmedAlready)
{
	// Don't let the user casually move/change important files/folders
	//
	// This is a cheap replacement for having a real UID support turned
	// on and not running as root all the time

	if (confirmedAlready && *confirmedAlready == kConfirmedAll)
		return true;

	if (!DirectoryMatchesOrContains(entry, B_BEOS_DIRECTORY)
		&& !DirectoryMatchesOrContains(entry, B_USER_DIRECTORY))
		// quick way out
		return true;

	const char *warning = NULL;
	bool requireOverride = true;

	if (DirectoryMatches(entry, B_BEOS_DIRECTORY))
		warning = "If you %s the beos folder, you won't be able to "
			"boot BeOS! Are you sure you want to do this? To %s the folder "
			"anyway, hold down the Shift key and click \"Do it\".";
	else if (DirectoryMatchesOrContains(entry, B_BEOS_SYSTEM_DIRECTORY))
		warning = "If you %s the system folder or its contents, you "
			"won't be able to boot BeOS! Are you sure you want to do this? "
			"To %s the system folder or its contents anyway, hold down "
			"the Shift key and click \"Do it\".";
	else if (DirectoryMatches(entry, B_USER_DIRECTORY)) {
		warning = "If you %s the home folder, BeOS may not "
			"behave properly! Are you sure you want to do this? "
			"To %s the home anyway, click \"Do it\".";
		requireOverride = false;
	} else if (DirectoryMatchesOrContains(entry, B_USER_CONFIG_DIRECTORY)
		|| DirectoryMatchesOrContains(entry, B_COMMON_SETTINGS_DIRECTORY)) {
		
		if (DirectoryMatchesOrContains(entry, "beos_mime", B_USER_SETTINGS_DIRECTORY)
			|| DirectoryMatchesOrContains(entry, "beos_mime", B_COMMON_SETTINGS_DIRECTORY)) {
			warning = "If you %s the mime settings, BeOS may not "
				"behave properly! Are you sure you want to do this? "
				"To %s the mime settings anyway, click \"Do it\".";
			requireOverride = false;
		} else if (DirectoryMatches(entry, B_USER_CONFIG_DIRECTORY)) {
			warning = "If you %s the config folder, BeOS may not "
				"behave properly! Are you sure you want to do this? "
				"To %s the config folder anyway, click \"Do it\".";
			requireOverride = false;
		} else if (DirectoryMatches(entry, B_USER_SETTINGS_DIRECTORY)
			|| DirectoryMatches(entry, B_COMMON_SETTINGS_DIRECTORY)) {
			warning = "If you %s the settings folder, BeOS may not "
				"behave properly! Are you sure you want to do this? "
				"To %s the settings folder anyway, click \"Do it\".";
			requireOverride = false;
		}
	}
	
	if (!warning)
		return true;

	if (dontAsk)
		return false;

	if (confirmedAlready && *confirmedAlready == kConfirmedHomeMove
		&& !requireOverride)
		// we already warned about moving home this time around
		return true;

	char buffer[256];
	sprintf(buffer, warning, action, action);

	if ((new OverrideAlert("", buffer, "Do it", (requireOverride ? B_SHIFT_KEY : 0),
		"Cancel", 0, NULL, 0, B_WIDTH_AS_USUAL, B_WARNING_ALERT))->Go() == 1) {
		if (confirmedAlready)
			*confirmedAlready = kNotConfirmed;
		return false;
	}
	
	if (confirmedAlready) {
		if (!requireOverride)
			*confirmedAlready = kConfirmedHomeMove;
		else
			*confirmedAlready = kConfirmedAll;
	}

	return true;
}

}	// namespace BPrivate
