#  Copyright 2001 by Donn Cave, Seattle, Washington, USA.
#  All rights reserved.  Permission to copy, modify and distribute this
#  material is hereby granted, without fee, provided that the above
#  copyright notice appear in all copies.
#
#  Extend IMAP4 class for status bar support.
#
import posix
import posixpath
import socket
import string
import sys

import imaplib

BUFSZ = 8192

class Socket:
	def __init__(self, host, port):
		self.socket = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
		self.socket.connect((host, port))
		self.fileinit()
	def _read(self, n = BUFSZ):
		v = self.socket.recv(n)
		if self.idbg:
			print >> sys.stderr, 'Socket._read', repr(v)
		return v
	def _write(self, t):
		if self.idbg:
			print >> sys.stderr, 'Socket._write', repr(t)
		return self.socket.send(t)
	def _close(self):
		return self.socket.close()
	def getpeername(self):
		return self.socket.getpeername()
	def fileno(self):
		return self.socket.fileno()

class Pipe:
	def __init__(self, host, port):
		self.inp, self.outp = self.rimap(host)
		self.fileinit()
	def _read(self, n = BUFSZ):
		v = posix.read(self.inp, n)
		if self.idbg:
			print >> sys.stderr, 'Pipe._read', repr(v)
		return v
	def _write(self, t):
		if self.idbg:
			print >> sys.stderr, 'Pipe._write', repr(t)
		return posix.write(self.outp, t)
	def _close(self):
		posix.close(self.inp)
		posix.close(self.outp)

class File:
	def fileinit(self):
		self.buffer = ''
		self.eof = 0
		self._setup()
		self.percentview = None
	def _setup(self):
		pass
	def send(self, t, x = 0):
		return self._write(t)
	def sendall(self, data):
		while data:
			n = self.send(data)
			data = data[n:]
	def read(self, size):
		want = size
		if self.buffer:
			r = self.buffer[:want]
			self.buffer = self.buffer[want:]
			want -= len(r)
		else:
			r = ''
		if self.eof:
			return r
		if want > 4096:
			pv = self.percentview
		else:
			pv = None
		if pv:
			pv(None)
			if want < size:
				pv(((size - want) * 100.0)/size)
		while want > 0:
			c = self._read()
			if not c:
				self.eof = 1
				break
			if len(c) <= want:
				f = c
			else:
				f = c[:want]
				self.buffer += c[want:]
			if pv:
				pv((len(f) * 100.0)/size)
			want -= len(f)
			r += f
		return r
	def write(self, t):
		return self._write(t)
	def readline(self):
		while 1:
			rl = self.buffer.split('\n', 1)
			if len(rl) > 1:
				self.buffer = rl[1]
				return rl[0] + '\n'
			if self.eof:
				r = self.buffer
				self.buffer = ''
				return r
			c = self._read()
			if not c:
				self.eof = 1
			else:
				self.buffer += c
	def close(self):
		return self._close()

class SocketFile(File, Socket):
	pass

class SSLFile(File, Socket):
	def _setup(self):
		import ssl
		self.ctx = ssl.context()
		self.client = ssl.client(self.socket.fileno(), self.ctx)
	def _write(self, t):
		return self.client.write(t)
	def _read(self, n = BUFSZ):
		return self.client.read(n)

class RshFile(File, Pipe):
	rsh = None
	def rimap(self, host):
		if not RshFile.rsh:
			for p in posix.environ['PATH'].split(':'):
				RshFile.rsh = p + '/rsh'
				if posixpath.exists(RshFile.rsh):
					break
			else:
				RshFile.rsh = '/usr/bin/rsh'
		rsh = RshFile.rsh
		user = None
		d0, d1 = posix.pipe()
		u0, u1 = posix.pipe()
		pid = posix.fork()
		if pid:
			posix.close(d0)
			posix.close(u1)
			return u0, d1
		posix.dup2(d0, 0)
		posix.close(d0)
		posix.close(d1)
		posix.close(u0)
		posix.dup2(u1, 1)
		posix.close(u1)
		try:
		#	posix.execve(rsh, ('rsh', '-l', user, host, '/etc/rimapd'), posix.environ)
			posix.execve(rsh, ('rsh', host, '/etc/rimapd'), posix.environ)
		finally:
			try:
				posix.write(2, 'exec %s failed\n' % self.rsh)
			except:
				pass
			posix._exit(117)

class IMAP4(imaplib.IMAP4):
	Connector = SocketFile
	def setpercentview(self, pv):
		self.file.percentview = pv
	def open(self, host, port):
		self.sock = self.Connector(host, port)
		self.file = self.sock
		self.file.idbg = 0
		imaplib.Commands['IDLE'] = imaplib.Commands['FETCH']
	def idle(self):
		name = 'IDLE'
		# typ, dat = self._simple_command(name)
		# return self.untagged_response(typ, dat, name)
		# --- stolen from IMAP4._command ---
		for typ in ('OK', 'NO', 'BAD'):
			if self.untagged_responses.has_key(typ):
				del self.untagged_responses[typ]
		tag = self._new_tag()
		self.sock.send('%s IDLE\r\n' % (tag,))
		# try:
		# 	self.sock.send('%s IDLE\r\n' % (tag,))
		# except socket.error, val:
		# 	raise self.abort('socket error: %s' % val)
		# ---
	def unidle(self):
		self.sock.send('DONE\r\n')
		self._get_response()

class SSLIMAP4(IMAP4):
	Connector = SSLFile

class RIMAP4(IMAP4):
	Connector = RshFile
