// driver.cpp
//
// Canon Bubble Jet Printers driver for BeOS

#include <stdio.h>
#include <string.h>

#include <StorageKit.h>

#include "driver.h"

#include "BJ130e.h"
#include "BJC.h"

typedef enum {
	UNKNOWN_PRINTER_CLASS,
	BJ130E_PRINTER_CLASS,
	BJC_PRINTER_CLASS,
	INVALID_PRINTER_CLASS = -1
} printer_class;

typedef struct {
	printer_class	class_id;
	char *			class_name;
} supported_printer_class;

supported_printer_class g_supported_printer_classes[] =
	{	{ BJ130E_PRINTER_CLASS, "bj130e" },
		{ BJC_PRINTER_CLASS,	"bjc" },
		{ INVALID_PRINTER_CLASS, NULL }
	};

static printer_class	get_printer_class(BNode * spool_dir);	
static PrinterDriver * 	instanciate_driver(BNode * spool_dir);


BMessage * take_job(BFile * spool_file, BNode * spool_dir, BMessage * msg)
{
	PrinterDriver * driver;
	
	driver = instanciate_driver(spool_dir);
	if ( driver->PrintJob(spool_file, spool_dir, msg) == B_OK )
		msg = new BMessage('okok');
	else	
		msg = new BMessage('baad');
	
	delete driver;
			
	return msg;
}



BMessage * config_page(BNode * spool_dir, BMessage * msg)
{
	BMessage *			pagesetup_msg;
	PrinterDriver * 	driver;

	pagesetup_msg = new BMessage(*msg);

	driver = instanciate_driver(spool_dir);
	if ( driver->PageSetup(pagesetup_msg) == B_OK )
		pagesetup_msg->what = ('okok');
	else
		{
		delete pagesetup_msg;
		pagesetup_msg = NULL;
		};
		
	delete driver;
	
	return pagesetup_msg;
}



BMessage * config_job(BNode * spool_dir, BMessage * msg)
{
	BMessage *			jobsetup_msg;
	PrinterDriver * 	driver;

	jobsetup_msg = new BMessage(*msg);

	driver = instanciate_driver(spool_dir);
	if ( driver->JobSetup(jobsetup_msg) == B_OK )
		jobsetup_msg->what = ('okok');
	else
		{
		delete jobsetup_msg;
		jobsetup_msg = NULL;
		};

	delete driver;
	
	return jobsetup_msg;
}



char * add_printer(char * printer_name)
{
	PrinterDriver *	driver;
	char *			pn;

	pn = printer_name;
	
	driver = new PrinterDriver();
	if ( driver->PrinterSetup(pn) != B_OK )
		pn = NULL;

	delete driver;

	return pn; 
}


static PrinterDriver * instanciate_driver
	(
	BNode *	spool_dir
	)
{
	printer_class pc;
	
	// Get model type from spool_dir node
	pc = get_printer_class(spool_dir);
	switch(pc)
		{
		case BJ130E_PRINTER_CLASS:
			return new BJ130e();
			
		case BJC_PRINTER_CLASS:
			return new BJC();

		default:
			return new PrinterDriver();
		};

}


static printer_class get_printer_class
	(
	BNode *	spool_dir
	)
{
	char	buffer[512];
	BPath * path;
	
	
	// first, find the printer_model name, and load its description file
	path = new BPath();
	
	// find printer model name of this printer 
	spool_dir->ReadAttr("printer_model", B_STRING_TYPE, 0, buffer, sizeof(buffer));
	
	find_directory(B_BEOS_ETC_DIRECTORY, path);
	path->Append("bubblejet");
	path->Append(buffer);

	// open the printer model description file
	
	FILE *	f;
	f = fopen(path->Path(), "rt");
	
	delete path;
	
	if ( !f )
		return UNKNOWN_PRINTER_CLASS;
	
	fgets(buffer, 512, f);	// read only first line
	fclose(f);
	
	char * token;

	token = strtok(buffer, " \t\n\r");
	if ( strcmp(token, "class") == 0 )
		{
		token = strtok(NULL, " \t\n\r");
		
		supported_printer_class	* spc;
		
		spc = g_supported_printer_classes;
		while(spc->class_name)
			{
			if ( strcmp(token, spc->class_name) == 0)
				return spc->class_id;
				
			spc++;
			}; 
		};

	return UNKNOWN_PRINTER_CLASS;		
}
